// ===================================================================
// SECURITY HELPERS - XSS Prevention & Input Sanitization
// ===================================================================

/**
 * Escape HTML to prevent XSS attacks
 * Converts special characters to HTML entities
 */
function escapeHtml(text) {
    if (typeof text !== 'string') {
        text = String(text);
    }

    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Create element safely with text content (no XSS)
 * @param {string} tag - HTML tag name
 * @param {object} attributes - Element attributes
 * @param {string} textContent - Text content (will be escaped)
 * @returns {HTMLElement}
 */
function createElementSafe(tag, attributes = {}, textContent = '') {
    const element = document.createElement(tag);

    // Set attributes
    for (const [key, value] of Object.entries(attributes)) {
        if (key === 'className') {
            element.className = value;
        } else if (key === 'onclick' || key.startsWith('on')) {
            // Never set event handlers via attributes (XSS risk)
            console.warn('⚠️ Event handlers should be added via addEventListener, not attributes');
        } else {
            element.setAttribute(key, value);
        }
    }

    // Set text content (safe from XSS)
    if (textContent) {
        element.textContent = textContent;
    }

    return element;
}

/**
 * Sanitize URL to prevent javascript: and data: URIs
 * @param {string} url - URL to sanitize
 * @returns {string} - Safe URL or empty string
 */
function sanitizeUrl(url) {
    if (!url || typeof url !== 'string') {
        return '';
    }

    const trimmed = url.trim().toLowerCase();

    // Block dangerous protocols
    if (trimmed.startsWith('javascript:') ||
        trimmed.startsWith('data:') ||
        trimmed.startsWith('vbscript:')) {
        console.warn('⚠️ Blocked dangerous URL:', url);
        return '';
    }

    // Allow http, https, chrome-extension
    if (trimmed.startsWith('http://') ||
        trimmed.startsWith('https://') ||
        trimmed.startsWith('chrome-extension://') ||
        trimmed.startsWith('/') ||
        trimmed.startsWith('./')) {
        return url;
    }

    // Default: prepend https://
    return 'https://' + url;
}

/**
 * Validate and sanitize username
 * @param {string} username - Username to validate
 * @returns {object} - {valid: boolean, sanitized: string, error: string}
 */
function validateUsername(username) {
    if (!username || typeof username !== 'string') {
        return { valid: false, sanitized: '', error: 'Username is required' };
    }

    const trimmed = username.trim();

    // Length check
    if (trimmed.length < 3) {
        return { valid: false, sanitized: trimmed, error: 'Username must be at least 3 characters' };
    }

    if (trimmed.length > 50) {
        return { valid: false, sanitized: trimmed, error: 'Username must be less than 50 characters' };
    }

    // Remove @ symbol if present (Fiverr usernames don't need it)
    let sanitized = trimmed.replace(/^@+/, '');

    // Allow only alphanumeric, underscore, hyphen
    if (!/^[a-zA-Z0-9_-]+$/.test(sanitized)) {
        return {
            valid: false,
            sanitized: sanitized,
            error: 'Username can only contain letters, numbers, underscore, and hyphen'
        };
    }

    return { valid: true, sanitized: sanitized, error: '' };
}

/**
 * Validate email address
 * @param {string} email - Email to validate
 * @returns {object} - {valid: boolean, sanitized: string, error: string}
 */
function validateEmail(email) {
    if (!email || typeof email !== 'string') {
        return { valid: false, sanitized: '', error: 'Email is required' };
    }

    const trimmed = email.trim().toLowerCase();

    // Basic email regex
    const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;

    if (!emailRegex.test(trimmed)) {
        return { valid: false, sanitized: trimmed, error: 'Invalid email format' };
    }

    if (trimmed.length > 100) {
        return { valid: false, sanitized: trimmed, error: 'Email is too long' };
    }

    return { valid: true, sanitized: trimmed, error: '' };
}

/**
 * Validate time interval (seconds)
 * @param {number|string} interval - Interval in seconds
 * @returns {object} - {valid: boolean, sanitized: number, error: string}
 */
function validateTimeInterval(interval) {
    const num = parseInt(interval);

    if (isNaN(num)) {
        return { valid: false, sanitized: 30, error: 'Invalid interval' };
    }

    if (num < 10) {
        return { valid: false, sanitized: num, error: 'Interval must be at least 10 seconds' };
    }

    if (num > 3600) {
        return { valid: false, sanitized: num, error: 'Interval must be less than 1 hour' };
    }

    return { valid: true, sanitized: num, error: '' };
}

/**
 * Sanitize text content (remove HTML tags)
 * @param {string} text - Text to sanitize
 * @returns {string} - Sanitized text
 */
function sanitizeText(text) {
    if (!text || typeof text !== 'string') {
        return '';
    }

    // Create element and use textContent to strip HTML
    const div = document.createElement('div');
    div.textContent = text;
    return div.textContent;
}

/**
 * Build toast notification safely
 * @param {string} title - Toast title
 * @param {string} message - Toast message
 * @param {string} type - Toast type (success, error, warning, info)
 * @returns {HTMLElement} - Toast element
 */
function buildToastElement(title, message, type = 'info') {
    // Icons as safe SVG strings
    const icons = {
        success: `<svg class="toast-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
      <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
      <polyline points="22 4 12 14.01 9 11.01"></polyline>
    </svg>`,
        error: `<svg class="toast-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
      <circle cx="12" cy="12" r="10"></circle>
      <line x1="15" y1="9" x2="9" y2="15"></line>
      <line x1="9" y1="9" x2="15" y2="15"></line>
    </svg>`,
        warning: `<svg class="toast-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
      <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path>
      <line x1="12" y1="9" x2="12" y2="13"></line>
      <line x1="12" y1="17" x2="12.01" y2="17"></line>
    </svg>`,
        info: `<svg class="toast-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round">
      <circle cx="12" cy="12" r="10"></circle>
      <line x1="12" y1="16" x2="12" y2="12"></line>
      <line x1="12" y1="8" x2="12.01" y2="8"></line>
    </svg>`
    };

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    // Create structure safely
    const iconWrapper = document.createElement('div');
    iconWrapper.innerHTML = icons[type] || icons.info; // SVG is safe

    const contentWrapper = document.createElement('div');
    contentWrapper.className = 'toast-content';

    const titleEl = document.createElement('div');
    titleEl.className = 'toast-title';
    titleEl.textContent = title; // Safe from XSS

    const messageEl = document.createElement('div');
    messageEl.className = 'toast-message';
    messageEl.textContent = message; // Safe from XSS

    const closeBtn = document.createElement('button');
    closeBtn.className = 'toast-close';
    closeBtn.textContent = '×';

    // Build structure
    contentWrapper.appendChild(titleEl);
    contentWrapper.appendChild(messageEl);

    toast.appendChild(iconWrapper.firstElementChild); // Append SVG element
    toast.appendChild(contentWrapper);
    toast.appendChild(closeBtn);

    return toast;
}

// Export for use in other scripts
if (typeof window !== 'undefined') {
    window.SecurityHelpers = {
        escapeHtml,
        createElementSafe,
        sanitizeUrl,
        validateUsername,
        validateEmail,
        validateTimeInterval,
        sanitizeText,
        buildToastElement
    };
}
