/**
 * Safe Message Sender with Type Whitelisting
 * Prevents unauthorized message types from being sent
 */

// ✅ Whitelist of allowed message types
const ALLOWED_MESSAGE_TYPES = [
    'registerUsername',
    'setInterval',
    'updateGigs',
    'setRefreshMode',
    'startBot',
    'stopBot',
    'newMessage',
    'getState',
    'syncConversationToMongoDB',
    'updateSummaryInMongoDB',
    'updateAnalyticsAfterReply',
    'generateReplyFull',
    'keepAlive',
    'verifyUsername',
    'clickInboxLink',
    'clickUnreadContact',
    'readAndReply',
    'checkMoreUnreads',
    'checkMessages',
    'sendReply',
    'getPageInfo',
    'forceStopBot',
    'restartAutomation',
    'reloadSelectors',
    'addLog',
    'updateAnalytics',
    'updateStats',
    'planExpired',
    'repliesLimitReached',
    'domMessageDetected'
];

/**
 * Safely send message to runtime with type validation
 * @param {Object} message - Message object with 'action' property
 * @param {Function} responseCallback - Optional callback for response
 * @returns {Promise} Promise that resolves with response or rejects with error
 */
function safeSendMessage(message, responseCallback = null) {
    return new Promise((resolve, reject) => {
        // Validate message object
        if (!message || typeof message !== 'object') {
            const error = new Error('Invalid message: must be an object');
            console.error('🚫 safeSendMessage:', error);
            if (responseCallback) responseCallback({ success: false, error: error.message });
            return reject(error);
        }

        // Validate action property
        if (!message.action || typeof message.action !== 'string') {
            const error = new Error('Invalid message: missing or invalid action property');
            console.error('🚫 safeSendMessage:', error);
            if (responseCallback) responseCallback({ success: false, error: error.message });
            return reject(error);
        }

        // ✅ Check if action is in whitelist
        if (!ALLOWED_MESSAGE_TYPES.includes(message.action)) {
            const error = new Error(`Rejected: message type '${message.action}' is not whitelisted`);
            console.error('🚫 safeSendMessage:', error);
            if (responseCallback) responseCallback({ success: false, error: error.message });
            return reject(error);
        }

        // ✅ Send message if validated
        try {
            chrome.runtime.sendMessage(message, (response) => {
                if (chrome.runtime.lastError) {
                    const error = new Error(chrome.runtime.lastError.message);
                    console.error('❌ safeSendMessage - Runtime error:', error);
                    if (responseCallback) responseCallback({ success: false, error: error.message });
                    return reject(error);
                }

                // Success
                if (responseCallback) responseCallback(response);
                resolve(response);
            });
        } catch (err) {
            console.error('❌ safeSendMessage - Exception:', err);
            if (responseCallback) responseCallback({ success: false, error: err.message });
            reject(err);
        }
    });
}

/**
 * Validate sender for incoming messages (for use in message listeners)
 * @param {Object} sender - Chrome runtime sender object
 * @param {Object} message - Message object
 * @returns {Boolean} True if sender is valid
 */
function validateSender(sender, message) {
    // Ensure sender is from this extension
    if (!sender.id || sender.id !== chrome.runtime.id) {
        console.warn('🚫 Invalid sender: not from this extension');
        return false;
    }

    // Validate message type
    if (!message || !message.action) {
        console.warn('🚫 Invalid message: missing action');
        return false;
    }

    if (!ALLOWED_MESSAGE_TYPES.includes(message.action)) {
        console.warn(`🚫 Invalid message type: ${message.action}`);
        return false;
    }

    return true;
}

// Make available globally
if (typeof window !== 'undefined') {
    window.safeSendMessage = safeSendMessage;
    window.validateSender = validateSender;
    window.ALLOWED_MESSAGE_TYPES = ALLOWED_MESSAGE_TYPES;
}
