// ===================================================================
// CONFIGURATION - Error Logging
// ===================================================================
const SHOW_ERRORS = false; // Set to true to show errors in console

// Override console.error and console.warn to suppress messages
if (!SHOW_ERRORS) {
  const originalError = console.error;
  const originalWarn = console.warn;

  console.error = function (...args) {
    // Suppress all error logs in production
    return;
  };

  console.warn = function (...args) {
    // Suppress all warning logs in production
    return;
  };
}

// ===================================================================
// GLOBAL ERROR HANDLERS - Prevent errors from showing in Chrome Extensions
// ===================================================================

// Catch unhandled promise rejections
self.addEventListener('unhandledrejection', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// Catch runtime errors
self.addEventListener('error', (event) => {
  if (!SHOW_ERRORS) {
    event.preventDefault(); // Prevent the error from appearing in Chrome's error log
  }
});

// ===================================================================
// Content script for Fiverr Auto Reply Bot
// This script runs on Fiverr pages to detect and reply to messages
// ===================================================================

console.log('Fiverr Auto Reply Bot content script loaded');

// Configuration
const API_BASE_URL = 'https://fivreply-production.up.railway.app/api';

// Default selectors (fallback if API fails)
const DEFAULT_SELECTORS = {
  messageContainer: '[data-message-id]',
  messageText: '.message-text, .message-body',
  replyButton: '[data-action="reply"]',
  replyTextarea: 'textarea[name="message"], textarea[placeholder*="message"]',
  sendButton: 'button[type="submit"], button[data-action="send"]',
  conversationList: '.conversation-item, .inbox-item',
  unreadBadge: '.unread-badge, .new-message-badge'
};

const config = {
  checkInterval: 5000,
  selectors: { ...DEFAULT_SELECTORS } // Will be replaced with dynamic selectors
};

// Dynamic selectors from database
let dynamicSelectors = null;
let selectorsLastLoaded = null;
const SELECTORS_CACHE_DURATION = 5 * 60 * 1000; // 5 minutes cache

// State
let lastMessageCount = 0;
let isProcessing = false;

// Load selectors from backend on script load
loadSelectorsFromBackend();

/**
 * Load selectors from backend API
 */
async function loadSelectorsFromBackend() {
  try {
    // Check if we have cached selectors
    const now = Date.now();
    if (dynamicSelectors && selectorsLastLoaded && (now - selectorsLastLoaded < SELECTORS_CACHE_DURATION)) {
      console.log('✅ Using cached selectors');
      return dynamicSelectors;
    }

    console.log('📡 Loading selectors from backend...');

    const response = await fetch(`${API_BASE_URL}/selectors?active=true`);
    const data = await response.json();

    if (data.success && data.selectors) {
      dynamicSelectors = data.selectors;
      selectorsLastLoaded = now;

      // Update config with new selectors
      updateConfigSelectors(data.selectors);

      console.log('✅ Selectors loaded from backend:', Object.keys(data.selectors).length, 'selectors');
      return dynamicSelectors;
    } else {
      console.warn('⚠️ Failed to load selectors from backend, using defaults');
      return DEFAULT_SELECTORS;
    }
  } catch (error) {
    console.error('❌ Error loading selectors from backend:', error);
    console.log('Using default selectors as fallback');
    return DEFAULT_SELECTORS;
  }
}

/**
 * Update config.selectors with dynamic selectors from backend
 */
function updateConfigSelectors(selectors) {
  // Map backend selector names to config names
  const mapping = {
    'message_container': 'messageContainer',
    'message_text': 'messageText',
    'reply_button': 'replyButton',
    'reply_textarea': 'replyTextarea',
    'send_button': 'sendButton',
    'conversation_list': 'conversationList',
    'unread_badge': 'unreadBadge'
  };

  for (const [backendName, configName] of Object.entries(mapping)) {
    if (selectors[backendName]) {
      config.selectors[configName] = selectors[backendName].value;
    }
  }

  console.log('✅ Config selectors updated');
}

/**
 * Get selector by name from dynamic selectors
 */
function getSelector(name) {
  if (dynamicSelectors && dynamicSelectors[name]) {
    return dynamicSelectors[name];
  }
  return null;
}

/**
 * Build selector array with fallback
 */
function buildSelectorArray(selectorNames) {
  const selectors = [];

  for (const name of selectorNames) {
    const selector = getSelector(name);
    if (selector) {
      selectors.push({
        type: selector.type,
        value: selector.value
      });

      // Add fallback if available
      if (selector.fallback) {
        selectors.push({
          type: selector.type,
          value: selector.fallback
        });
      }
    }
  }

  return selectors;
}

// Listen for messages from background script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Content script received message:', message);

  switch (message.action) {
    case 'reloadSelectors':
      // Force reload selectors from backend
      selectorsLastLoaded = null; // Clear cache
      loadSelectorsFromBackend()
        .then(() => sendResponse({ success: true, message: 'Selectors reloaded' }))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true;

    case 'verifyUsername':
      verifyUsername(message.expectedUsername)
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ error: error.message, verified: false }));
      return true;

    case 'clickInboxLink':
      clickInboxLink()
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true;

    case 'clickUnreadContact':
      clickUnreadContact()
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true;

    case 'readAndReply':
      readAndReply(message.apiKey, message.userGigs || [])
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true;

    case 'checkMoreUnreads':
      checkMoreUnreads()
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true;

    case 'checkMessages':
      checkForNewMessages()
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ error: error.message }));
      return true;

    case 'sendReply':
      sendReply(message.conversationId, message.reply)
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ error: error.message }));
      return true;

    case 'getPageInfo':
      sendResponse(getPageInfo());
      break;

    case 'forceStopBot':
      // Force stop all bot activities
      console.log('🛑 FORCE STOP - Halting all bot activities');
      isProcessing = false;

      // Clear any pending timeouts/intervals
      if (window.botActivityInterval) {
        clearInterval(window.botActivityInterval);
        window.botActivityInterval = null;
      }

      // Send log to popup
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'WARNING',
        message: '🛑 Bot stopped - All activities halted on this page'
      }).catch(() => { });

      sendResponse({ success: true, message: 'Bot stopped successfully' });
      break;

    case 'restartAutomation':
      // ✨ NEW: Restart automation flow from beginning
      console.log('🔄 RESTART - Restarting automation flow on seller dashboard...');

      // Reset processing flag to allow new automation cycle
      isProcessing = false;

      // Clear any pending operations
      if (window.botActivityInterval) {
        clearInterval(window.botActivityInterval);
        window.botActivityInterval = null;
      }

      // Get current page info
      const pageInfo = getPageInfo();
      console.log('📍 Current page:', pageInfo.currentPage);

      // Store gigs if provided
      if (message.gigs && message.gigs.length > 0) {
        console.log(`📦 Received ${message.gigs.length} gig(s) for automation restart`);
      }

      // Send log to popup
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'SUCCESS',
        message: '🔄 Automation cycle restarted on seller dashboard'
      }).catch(() => { });

      // Send success response
      sendResponse({
        success: true,
        message: 'Automation restarted successfully',
        currentPage: pageInfo.currentPage
      });

      console.log('✅ Automation restart complete - ready for next cycle');
      break;

    default:
      sendResponse({ error: 'Unknown action' });
  }
});

// Check for new messages on the page
async function checkForNewMessages() {
  if (isProcessing) {
    return { newMessages: 0, status: 'busy' };
  }

  try {
    isProcessing = true;

    // Check if we're on a Fiverr messages page
    const currentUrl = window.location.href;

    if (currentUrl.includes('/inbox') || currentUrl.includes('/messages')) {
      // Get unread message count
      const unreadBadges = document.querySelectorAll(config.selectors.unreadBadge);
      const newMessageCount = unreadBadges.length;

      if (newMessageCount > lastMessageCount) {
        const diff = newMessageCount - lastMessageCount;
        lastMessageCount = newMessageCount;

        // Process new messages
        await processNewMessages();

        return { newMessages: diff, status: 'processed' };
      }

      lastMessageCount = newMessageCount;
      return { newMessages: 0, status: 'no_new_messages' };
    }

    return { newMessages: 0, status: 'not_inbox_page' };

  } catch (error) {
    console.error('Error checking messages:', error);
    return { error: error.message };
  } finally {
    isProcessing = false;
  }
}

// Process new messages
async function processNewMessages() {
  try {
    // Find all conversation items
    const conversations = document.querySelectorAll(config.selectors.conversationList);

    for (const conversation of conversations) {
      // Check if conversation has unread messages
      const unreadBadge = conversation.querySelector(config.selectors.unreadBadge);

      if (unreadBadge) {
        // Click on conversation to open it
        conversation.click();

        // Wait for messages to load
        await sleep(1000);

        // Extract message details
        const messageData = extractMessageData();

        if (messageData) {
          // Send to background script for AI processing
          chrome.runtime.sendMessage({
            action: 'newMessage',
            data: messageData
          });

          // Wait before processing next conversation
          await sleep(2000);
        }
      }
    }
  } catch (error) {
    console.error('Error processing new messages:', error);
  }
}

// Extract message data from current conversation
function extractMessageData() {
  try {
    // Find all messages in the conversation
    const messages = document.querySelectorAll(config.selectors.messageContainer);

    if (messages.length === 0) {
      return null;
    }

    // Get the last message (most recent)
    const lastMessage = messages[messages.length - 1];
    const messageTextEl = lastMessage.querySelector(config.selectors.messageText);

    if (!messageTextEl) {
      return null;
    }

    const messageText = messageTextEl.textContent.trim();

    // Get conversation context (previous messages)
    const context = [];
    for (let i = Math.max(0, messages.length - 5); i < messages.length - 1; i++) {
      const msgEl = messages[i].querySelector(config.selectors.messageText);
      if (msgEl) {
        context.push(msgEl.textContent.trim());
      }
    }

    // Get conversation ID (try to extract from URL or data attributes)
    const conversationId = extractConversationId();

    return {
      conversationId: conversationId,
      message: messageText,
      context: context,
      timestamp: Date.now()
    };

  } catch (error) {
    console.error('Error extracting message data:', error);
    return null;
  }
}

// Extract conversation ID from page
function extractConversationId() {
  // Try to get from URL
  const urlMatch = window.location.href.match(/\/conversation\/([^\/\?]+)/);
  if (urlMatch) {
    return urlMatch[1];
  }

  // Try to get from data attributes
  const conversationEl = document.querySelector('[data-conversation-id]');
  if (conversationEl) {
    return conversationEl.getAttribute('data-conversation-id');
  }

  // Fallback to timestamp-based ID
  return `conv_${Date.now()}`;
}

/**
 * Type reply text into the message textarea
 * This function safely sets text in Fiverr's textarea and triggers necessary events
 */
async function typeReply(replyText) {
  console.log('🔍 Looking for reply textarea...');

  try {
    // Try multiple selectors to find the textarea
    const selectors = [
      'textarea[placeholder*="Type"]',
      'textarea[placeholder*="message"]',
      'textarea[placeholder*="reply"]',
      'textarea',
      '[contenteditable="true"]'
    ];

    let textarea = null;
    let selectorUsed = '';

    for (const selector of selectors) {
      textarea = document.querySelector(selector);
      if (textarea) {
        selectorUsed = selector;
        console.log(`✅ Found textarea with selector: ${selector}`);
        break;
      }
    }

    if (!textarea) {
      console.error('❌ Reply textarea not found with any selector');
      return {
        success: false,
        error: 'Reply textarea not found'
      };
    }

    console.log(`📝 Typing ${replyText.length} characters into textarea...`);

    // Method 1: For regular textarea elements
    if (textarea.tagName === 'TEXTAREA' || textarea.tagName === 'INPUT') {
      // Store original value
      const originalValue = textarea.value;

      // Set the value
      textarea.value = replyText;

      // Trigger all necessary events for React/Vue to detect the change
      const events = [
        new Event('input', { bubbles: true, cancelable: true }),
        new Event('change', { bubbles: true, cancelable: true }),
        new KeyboardEvent('keydown', { bubbles: true, cancelable: true, key: 'a' }),
        new KeyboardEvent('keyup', { bubbles: true, cancelable: true, key: 'a' }),
        new InputEvent('beforeinput', { bubbles: true, cancelable: true }),
        new InputEvent('input', { bubbles: true, cancelable: true, data: replyText })
      ];

      // Dispatch all events
      events.forEach(event => {
        try {
          textarea.dispatchEvent(event);
        } catch (e) {
          // Some events might fail, that's OK
        }
      });

      // Set React's internal state if available
      if (textarea._valueTracker) {
        textarea._valueTracker.setValue(originalValue);
      }

      console.log('✅ Text typed successfully using textarea method');
    }
    // Method 2: For contenteditable divs
    else if (textarea.getAttribute('contenteditable') === 'true') {
      // Clear existing content
      textarea.textContent = '';

      // Insert text
      textarea.textContent = replyText;

      // Trigger events
      textarea.dispatchEvent(new Event('input', { bubbles: true }));
      textarea.dispatchEvent(new Event('change', { bubbles: true }));

      console.log('✅ Text typed successfully using contenteditable method');
    }

    // Wait for UI to update
    await sleep(500);

    // Verify the text was set
    const currentValue = textarea.value || textarea.textContent || '';
    if (currentValue !== replyText) {
      console.warn(`⚠️ Text verification failed. Expected ${replyText.length} chars, got ${currentValue.length} chars`);
      console.warn(`Expected: "${replyText.substring(0, 50)}..."`);
      console.warn(`Got: "${currentValue.substring(0, 50)}..."`);
    } else {
      console.log('✅ Text verification passed - content matches');
    }

    return {
      success: true,
      message: 'Reply typed successfully',
      selectorUsed: selectorUsed,
      charsTyped: replyText.length
    };

  } catch (error) {
    console.error('❌ Error typing reply:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Click the send button to send the reply
 */
async function clickSendButton() {
  console.log('🔍 Looking for send button...');

  try {
    // Try multiple selectors to find the send button
    const selectors = [
      'button[type="submit"]',
      'button:has(svg)',
      'button[aria-label*="Send"]',
      'button[aria-label*="send"]',
      'button.send',
      '[data-action="send"]',
      'button:last-of-type'
    ];

    let sendButton = null;
    let selectorUsed = '';

    for (const selector of selectors) {
      const buttons = document.querySelectorAll(selector);

      // Find a button that looks like a send button
      for (const btn of buttons) {
        const text = btn.textContent.toLowerCase();
        const ariaLabel = btn.getAttribute('aria-label')?.toLowerCase() || '';

        if (text.includes('send') ||
          ariaLabel.includes('send') ||
          btn.querySelector('svg') ||
          btn.type === 'submit') {
          sendButton = btn;
          selectorUsed = selector;
          break;
        }
      }

      if (sendButton) {
        console.log(`✅ Found send button with selector: ${selector}`);
        break;
      }
    }

    if (!sendButton) {
      console.error('❌ Send button not found with any selector');
      return {
        success: false,
        error: 'Send button not found'
      };
    }

    console.log('🖱️ Clicking send button...');

    // Click the button
    sendButton.click();

    // Wait for the message to be sent
    await sleep(1000);

    console.log('✅ Send button clicked successfully');

    return {
      success: true,
      message: 'Send button clicked successfully',
      selectorUsed: selectorUsed
    };

  } catch (error) {
    console.error('❌ Error clicking send button:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

// Send reply to a conversation
async function sendReply(conversationId, replyText) {
  try {
    // Find the reply textarea
    const textarea = document.querySelector(config.selectors.replyTextarea);

    if (!textarea) {
      throw new Error('Reply textarea not found');
    }

    // Set the reply text
    textarea.value = replyText;

    // Trigger input event to update React/Vue state if used
    const inputEvent = new Event('input', { bubbles: true });
    textarea.dispatchEvent(inputEvent);

    // Wait a bit for the UI to update
    await sleep(500);

    // Find and click the send button
    const sendButton = document.querySelector(config.selectors.sendButton);

    if (!sendButton) {
      throw new Error('Send button not found');
    }

    sendButton.click();

    console.log('Reply sent:', replyText);

    return { success: true, conversationId, replyText };

  } catch (error) {
    console.error('Error sending reply:', error);
    return { success: false, error: error.message };
  }
}

// Get current page information
function getPageInfo() {
  return {
    url: window.location.href,
    title: document.title,
    isInboxPage: window.location.href.includes('/inbox') || window.location.href.includes('/messages'),
    messageCount: document.querySelectorAll(config.selectors.messageContainer).length,
    unreadCount: document.querySelectorAll(config.selectors.unreadBadge).length
  };
}

// Utility: Sleep function
function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * Report selector success to backend (for tracking)
 */
async function reportSelectorSuccess(selectorName) {
  try {
    await fetch(`${API_BASE_URL}/selectors/${selectorName}/test`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ success: true })
    });
    console.log(`✅ Reported selector success: ${selectorName}`);
  } catch (error) {
    console.warn('Failed to report selector success:', error.message);
  }
}

/**
 * Report selector failure to backend (for tracking)
 */
async function reportSelectorFailure(selectorName) {
  try {
    await fetch(`${API_BASE_URL}/selectors/${selectorName}/test`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ success: false })
    });
    console.log(`❌ Reported selector failure: ${selectorName}`);
  } catch (error) {
    console.warn('Failed to report selector failure:', error.message);
  }
}

/**
 * Extract user ID from current page URL
 * Fiverr inbox URLs: https://www.fiverr.com/inbox/username
 */
function extractUserIdFromURL() {
  try {
    const url = window.location.href;
    const match = url.match(/\/inbox\/([a-zA-Z0-9_-]+)/);
    if (match && match[1]) {
      console.log(`👤 Extracted user ID from URL: ${match[1]}`);
      return match[1];
    }
    return null;
  } catch (error) {
    console.error('Error extracting user ID:', error);
    return null;
  }
}

// Monitor DOM changes for new messages
const observer = new MutationObserver((mutations) => {
  // Check if new messages were added
  const hasNewMessages = mutations.some(mutation => {
    return Array.from(mutation.addedNodes).some(node => {
      if (node.nodeType === 1) { // Element node
        return node.matches?.(config.selectors.messageContainer) ||
          node.querySelector?.(config.selectors.messageContainer);
      }
      return false;
    });
  });

  if (hasNewMessages) {
    console.log('New message detected via DOM mutation');
    // Notify background script
    chrome.runtime.sendMessage({
      action: 'domMessageDetected'
    }).catch(() => {
      // Background script might be busy
    });
  }
});

// Start observing when DOM is ready
function startObserving() {
  const targetNode = document.body;

  if (targetNode) {
    observer.observe(targetNode, {
      childList: true,
      subtree: true
    });

    console.log('Started observing DOM for new messages');
  }
}

// Initialize
function init() {
  console.log('Initializing Fiverr Auto Reply content script');

  // Start observing after a short delay to let the page load
  setTimeout(() => {
    startObserving();
  }, 2000);

  // Log page info
  console.log('Page info:', getPageInfo());
}

// Wait for page to be fully loaded
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

// Add visual indicator that extension is active
function addExtensionIndicator() {
  const indicator = document.createElement('div');
  indicator.id = 'fiverr-auto-reply-indicator';
  indicator.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: linear-gradient(135deg, #22c55e, #16a34a);
    color: white;
    padding: 10px 16px;
    border-radius: 8px;
    font-size: 12px;
    font-weight: 600;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    z-index: 999999;
    display: flex;
    align-items: center;
    gap: 8px;
  `;
  indicator.innerHTML = `
    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
      <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
    </svg>
    <span>Auto Reply Active</span>
  `;

  document.body.appendChild(indicator);

  // Remove after 3 seconds
  setTimeout(() => {
    indicator.style.transition = 'opacity 0.5s';
    indicator.style.opacity = '0';
    setTimeout(() => indicator.remove(), 500);
  }, 3000);
}

// Show indicator when script loads
setTimeout(addExtensionIndicator, 1000);

// ============================================================================
// BOT AUTOMATION FUNCTIONS
// ============================================================================

/**
 * Verify Fiverr username on dashboard
 */
async function verifyUsername(expectedUsername) {
  console.log('Verifying username:', expectedUsername);

  try {
    await sleep(2000);

    const xpath = "//p[@class='rbwcxmk _1ljyzmr1f4 _1ljyzmr1cm _1ljyzmr6 _1ljyzmr2']";

    const usernameElement = document.evaluate(
      xpath,
      document,
      null,
      XPathResult.FIRST_ORDERED_NODE_TYPE,
      null
    ).singleNodeValue;

    // Try detecting seller name using user provided XPath with retries
    let sellerName = null;
    try {
      const sellerNameXpath = "//p[@class='rbwcxmk _1ljyzmr1fe _1ljyzmr1cw _1ljyzmr8 _1ljyzmr2']";

      // Retry logic for seller name
      for (let i = 0; i < 3; i++) {
        const sellerNameElement = document.evaluate(
          sellerNameXpath,
          document,
          null,
          XPathResult.FIRST_ORDERED_NODE_TYPE,
          null
        ).singleNodeValue;

        if (sellerNameElement) {
          sellerName = sellerNameElement.textContent.trim();
          console.log(`✅ Found seller name (attempt ${i + 1}): "${sellerName}"`);
          break;
        } else {
          console.log(`⚠️ Seller name element not found (attempt ${i + 1})...`);
          if (i < 2) await sleep(1000); // Wait before retry
        }
      }
    } catch (e) {
      console.log('Error extracting seller name:', e);
    }

    if (!usernameElement) {
      const allText = document.body.innerText;
      const usernameMatch = allText.match(/@(\w+)/);

      if (usernameMatch && usernameMatch[1]) {
        const foundUsername = usernameMatch[1];
        const cleanExpected = expectedUsername.replace(/^@/, '').toLowerCase().trim();
        const verified = foundUsername.toLowerCase() === cleanExpected;

        return {
          verified: verified,
          foundUsername: foundUsername,
          expectedUsername: cleanExpected,
          method: 'alternative',
          sellerName: sellerName // Return seller name if found
        };
      }

      return {
        verified: false,
        foundUsername: null,
        error: 'Username element not found on page.'
      };
    }

    let foundUsernameRaw = usernameElement.textContent.trim();
    let foundUsername = foundUsernameRaw;
    const atMatch = foundUsernameRaw.match(/@(\w+)/);
    if (atMatch && atMatch[1]) {
      foundUsername = atMatch[1];
    }

    const cleanExpected = expectedUsername.replace(/^@/, '').toLowerCase().trim();
    const verified = foundUsername.toLowerCase() === cleanExpected;

    if (verified) {
      showVerificationIndicator(true, foundUsername);
    } else {
      showVerificationIndicator(false, foundUsername);
    }

    return {
      verified: verified,
      foundUsername: foundUsername,
      expectedUsername: cleanExpected,
      method: 'xpath',
      sellerName: sellerName // Return seller name
    };

  } catch (error) {
    console.error('Error verifying username:', error);
    return {
      verified: false,
      error: error.message
    };
  }
}

/**
 * Show verification indicator on page
 */
function showVerificationIndicator(success, username) {
  const existing = document.getElementById('fiverr-username-verification');
  if (existing) {
    existing.remove();
  }

  const indicator = document.createElement('div');
  indicator.id = 'fiverr-username-verification';
  indicator.style.cssText = `
    position: fixed;
    top: 20px;
    left: 50%;
    transform: translateX(-50%);
    background: ${success ? 'linear-gradient(135deg, #22c55e, #16a34a)' : 'linear-gradient(135deg, #ef4444, #dc2626)'};
    color: white;
    padding: 16px 24px;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 600;
    box-shadow: 0 8px 24px rgba(0,0,0,0.3);
    z-index: 999999;
    display: flex;
    align-items: center;
    gap: 12px;
    animation: slideDown 0.5s ease-out;
  `;

  indicator.innerHTML = `
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
      ${success
      ? '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline>'
      : '<circle cx="12" cy="12" r="10"></circle><line x1="15" y1="9" x2="9" y2="15"></line><line x1="9" y1="9" x2="15" y2="15"></line>'
    }
    </svg>
    <span>${success ? `✓ Username Verified: ${username}` : `✗ Username Mismatch: ${username}`}</span>
  `;

  document.body.appendChild(indicator);

  const style = document.createElement('style');
  style.textContent = `
    @keyframes slideDown {
      from {
        transform: translateX(-50%) translateY(-100px);
        opacity: 0;
      }
      to {
        transform: translateX(-50%) translateY(0);
        opacity: 1;
      }
    }
  `;
  document.head.appendChild(style);

  setTimeout(() => {
    indicator.style.transition = 'opacity 0.5s, transform 0.5s';
    indicator.style.opacity = '0';
    indicator.style.transform = 'translateX(-50%) translateY(-100px)';
    setTimeout(() => indicator.remove(), 500);
  }, 5000);
}

/**
 * Click inbox link to navigate to inbox page
 */
async function clickInboxLink() {
  console.log('Attempting to click inbox link...');

  try {
    const inboxLinkXPath = "//a[@href='/inbox']";

    const inboxLink = document.evaluate(
      inboxLinkXPath,
      document,
      null,
      XPathResult.FIRST_ORDERED_NODE_TYPE,
      null
    ).singleNodeValue;

    if (!inboxLink) {
      return {
        success: false,
        error: 'Inbox link not found on page'
      };
    }

    console.log('Inbox link found, clicking...');
    inboxLink.click();

    return {
      success: true,
      message: 'Inbox link clicked successfully'
    };

  } catch (error) {
    console.error('Error clicking inbox link:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Find and click first unread contact
 */
async function clickUnreadContact() {
  console.log('🔍 Attempting to find and click unread contact...');
  console.log('Current URL:', window.location.href);

  try {
    // Try to load fresh selectors
    await loadSelectorsFromBackend();

    // Build unread contact selectors from database
    const unreadSelectorNames = [
      'unread_contact_first',
      'unread_contact_regular',
      'unread_contact_general'
    ];

    const unreadSelectors = buildSelectorArray(unreadSelectorNames);

    // Fallback to hardcoded if no dynamic selectors
    if (unreadSelectors.length === 0) {
      console.log('⚠️ Using fallback hardcoded selectors');
      unreadSelectors.push(
        {
          type: 'xpath',
          value: "//div[@class='_1v0154n0 contact first'][.//div[@class='_1hjuhhy14 _1hjuhhy19 _1hjuhhy1j qem7dds _1v0154n0 _1v0154n10j _1v0154n1j _1v0154n73 _1v0154nzu _1v0154n17c _1v0154n195 _1v0154n25m _1v0154n1b8']]"
        },
        {
          type: 'xpath',
          value: "//div[@class='_1v0154n0 contact'][.//div[@class='_1hjuhhy14 _1hjuhhy19 _1hjuhhy1j qem7dds _1v0154n0 _1v0154n10j _1v0154n1j _1v0154n73 _1v0154nzu _1v0154n17c _1v0154n195 _1v0154n25m _1v0154n1b8']]"
        },
        {
          type: 'xpath',
          value: "//div[contains(@class, 'contact')][.//p[contains(@class, 'a17q931eu') and normalize-space(text()) != '']]"
        }
      );
    }

    for (let i = 0; i < unreadSelectors.length; i++) {
      const selectorObj = unreadSelectors[i];
      console.log(`Trying unread selector ${i + 1}/${unreadSelectors.length} (${selectorObj.type})...`);

      try {
        let unreadContact = null;

        if (selectorObj.type === 'xpath') {
          unreadContact = document.evaluate(
            selectorObj.value,
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
          ).singleNodeValue;
        } else if (selectorObj.type === 'css') {
          unreadContact = document.querySelector(selectorObj.value);
        } else if (selectorObj.type === 'id') {
          unreadContact = document.getElementById(selectorObj.value);
        } else if (selectorObj.type === 'class') {
          unreadContact = document.querySelector(`.${selectorObj.value}`);
        }

        if (unreadContact) {
          console.log(`✅ Unread contact found with selector ${i + 1} (${selectorObj.type})`);
          unreadContact.click();

          // Report selector success to backend
          reportSelectorSuccess(unreadSelectorNames[i] || `selector_${i}`);

          return {
            success: true,
            message: 'Unread contact clicked successfully',
            method: `selector_${i + 1}`,
            selectorType: selectorObj.type
          };
        }
      } catch (error) {
        console.warn(`Selector ${i + 1} (${selectorObj.type}) failed:`, error.message);
        // Report selector failure to backend
        reportSelectorFailure(unreadSelectorNames[i] || `selector_${i}`);
        continue;
      }
    }

    return {
      success: false,
      error: 'No unread contacts found'
    };

  } catch (error) {
    console.error('Error in clickUnreadContact:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Extract gig link from conversation page using XPath
 */
function extractGigLink() {
  console.log('🔍 Extracting gig link from conversation page...');
  console.log('📍 Current URL:', window.location.href);

  try {
    // Strategy 1: XPath - Find any link with gig pattern (//a[@href='/username/gig-slug'])
    console.log('Strategy 1: Using XPath to find gig links...');

    // XPath to find all links starting with / and containing hyphens (typical gig pattern)
    const xpathQuery = "//a[starts-with(@href, '/') and contains(@href, '-')]";

    const result = document.evaluate(
      xpathQuery,
      document,
      null,
      XPathResult.ORDERED_NODE_SNAPSHOT_TYPE,
      null
    );

    console.log(`   Found ${result.snapshotLength} potential gig links via XPath`);

    for (let i = 0; i < result.snapshotLength; i++) {
      const link = result.snapshotItem(i);
      const href = link.getAttribute('href');

      // Validate it's a gig link pattern: /username/gig-slug
      // Example: /kerry_hoke/create-60-monetizable-motivational-videos-for-youtube
      if (href && href.match(/^\/[a-zA-Z0-9_-]+\/[a-zA-Z0-9_-]+/)) {
        // Exclude common non-gig links
        if (!href.includes('/inbox') &&
          !href.includes('/dashboard') &&
          !href.includes('/settings') &&
          !href.includes('/messages') &&
          !href.includes('/orders') &&
          !href.includes('/gigs')) {

          console.log('✅ Found gig link (XPath Strategy):', href);
          console.log('   Link text:', link.textContent.trim().substring(0, 50));
          console.log('   Full XPath:', `//a[@href='${href}']`);
          return href;
        }
      }
    }

    // Strategy 2: Fallback - CSS selector for any link with gig pattern
    console.log('Strategy 2: Fallback CSS selector search...');
    const allLinks = document.querySelectorAll('a[href^="/"]');
    console.log(`   Found ${allLinks.length} links starting with /`);

    for (const link of allLinks) {
      const href = link.getAttribute('href');

      // Match Fiverr gig URL pattern: /username/gig-title with hyphens
      if (href && href.includes('-') && href.match(/^\/[a-zA-Z0-9_-]+\/[a-zA-Z0-9_-]+/)) {
        // Exclude common non-gig links
        if (!href.includes('/inbox') &&
          !href.includes('/dashboard') &&
          !href.includes('/settings') &&
          !href.includes('/messages') &&
          !href.includes('/orders') &&
          !href.includes('/gigs')) {

          const parts = href.split('/').filter(p => p);
          if (parts.length === 2 && parts[1].includes('-')) {
            console.log('✅ Found gig link (CSS Fallback):', href);
            console.log('   Link text:', link.textContent.trim().substring(0, 50));
            return href;
          }
        }
      }
    }

    // Strategy 3: Look in message containers specifically
    console.log('Strategy 3: Searching in message containers...');
    const messageContainers = document.querySelectorAll('[class*="message"], [class*="content"], [class*="attachment"]');
    console.log(`   Found ${messageContainers.length} message containers`);

    for (const container of messageContainers) {
      const containerLinks = container.querySelectorAll('a[href^="/"]');
      for (const link of containerLinks) {
        const href = link.getAttribute('href');
        if (href && href.includes('-') && href.match(/^\/[a-zA-Z0-9_-]+\/[a-zA-Z0-9_-]+/)) {
          const parts = href.split('/').filter(p => p);
          if (parts.length === 2 && !href.includes('/inbox') && !href.includes('/dashboard')) {
            console.log('✅ Found gig link (Container Search):', href);
            console.log('   Link text:', link.textContent.trim().substring(0, 50));
            return href;
          }
        }
      }
    }

    console.log('⚠️ No gig link detected on this page');
    console.log('💡 This conversation may not be about a specific gig');
    return null;

  } catch (error) {
    console.error('❌ Error extracting gig link:', error);
    return null;
  }
}

/**
 * Read message content from conversation (IMPROVED VERSION)
 */
async function readMessageContent() {
  console.log('📖 Reading message content from conversation...');

  try {
    // ✨ NEW: Primary extraction using the new selector from div.content
    console.log('🔍 Using new message extraction with XPath selector...');

    // Primary selector: //div[contains(@class,'message') and .//p[contains(@class,'a17q931cm')]]
    const messageXPath = "//div[contains(@class,'message') and .//p[contains(@class,'a17q931cm')]]";

    console.log(`📍 XPath: ${messageXPath}`);

    const result = document.evaluate(
      messageXPath,
      document,
      null,
      XPathResult.ORDERED_NODE_SNAPSHOT_TYPE,
      null
    );

    console.log(`Found ${result.snapshotLength} message elements`);

    if (result.snapshotLength > 0) {
      const messages = [];
      const structuredMessages = [];
      let customerName = null;
      let customerCount = 0;
      let sellerCount = 0;

      for (let i = 0; i < result.snapshotLength; i++) {
        const messageElement = result.snapshotItem(i);

        // Extract sender name - look for p with class 'a17q931cm'
        const senderElement = messageElement.querySelector("p[class*='a17q931cm']");
        let sender = senderElement ? senderElement.textContent.trim() : 'Unknown';

        // Validate sender - skip if it's a warning/system message
        const invalidSenders = [
          'for added safety',
          'keep payments',
          'learn more',
          'fiverr',
          'system',
          'notice',
          'warning'
        ];

        const senderLower = sender.toLowerCase();
        const isInvalidSender = invalidSenders.some(invalid => senderLower.includes(invalid));

        if (isInvalidSender || sender.length > 50) {
          // This is not a real sender name, skip this element
          console.warn(`⚠️ Skipping invalid sender: "${sender.substring(0, 50)}..."`);
          continue;
        }

        // Extract message content - look for the actual message text
        let messageContent = '';

        // Try to find the content paragraph (usually has different class than sender)
        const contentElements = messageElement.querySelectorAll('p');
        for (const elem of contentElements) {
          const elemClass = elem.className || '';
          // Skip the sender name element
          if (!elemClass.includes('a17q931cm')) {
            const text = elem.textContent?.trim();
            if (text && text !== sender && text.length > 0) {
              messageContent = text;
              break;
            }
          }
        }

        // If still no content, try getting all text and remove sender name
        if (!messageContent) {
          const allText = messageElement.textContent?.trim() || '';
          messageContent = allText.replace(sender, '').trim();
        }

        if (!messageContent || messageContent.length < 2) {
          console.warn(`⚠️ No valid content found for message ${i + 1}`);
          continue;
        }

        // Remove timestamp patterns from message content
        // Common patterns: "Dec 06, 7:31 AM", "12/06, 7:31 PM", etc.
        messageContent = messageContent.replace(/^[A-Z][a-z]{2}\s+\d{1,2},\s+\d{1,2}:\d{2}\s+[AP]M/i, '').trim();
        messageContent = messageContent.replace(/^\d{1,2}\/\d{1,2},\s+\d{1,2}:\d{2}\s+[AP]M/i, '').trim();
        messageContent = messageContent.replace(/^\d{1,2}:\d{2}\s+[AP]M/i, '').trim();

        // If content is still too short after cleaning, skip
        if (messageContent.length < 2) {
          console.warn(`⚠️ Content too short after cleaning: "${messageContent}"`);
          continue;
        }

        // Determine role (user vs assistant)
        // Check if this is from "Me" (seller) or from customer
        const isSellerMessage = sender.toLowerCase() === 'me' ||
          messageElement.classList.contains('seller') ||
          messageElement.classList.contains('outgoing') ||
          messageElement.querySelector('[data-sender="seller"]');

        const role = isSellerMessage ? 'assistant' : 'user';

        // Track customer name (first non-"Me" sender we encounter)
        if (role === 'user' && !customerName && sender.toLowerCase() !== 'me') {
          customerName = sender;
        }

        // Count messages by role
        if (role === 'user') {
          customerCount++;
        } else {
          sellerCount++;
        }

        // Add to arrays
        messages.push(messageContent);
        structuredMessages.push({
          text: messageContent,
          content: messageContent,
          role: role,
          sender: sender,
          isCustomer: role === 'user',
          originalIndex: i,
          timestamp: new Date().toISOString()
        });

        console.log(`   ${i + 1}. [${role === 'user' ? '👤' : '🤝'}] ${sender}: "${messageContent.substring(0, 50)}${messageContent.length > 50 ? '...' : ''}"`);
      }

      if (messages.length > 0) {
        console.log(`✅ Successfully extracted ${messages.length} messages`);
        console.log(`   - Customer (${customerName || 'Unknown'}): ${customerCount} messages`);
        console.log(`   - Seller (Me): ${sellerCount} messages`);

        // Create formatted array for logging
        const formattedArray = structuredMessages.map(msg =>
          `${msg.sender}: ${msg.content}`
        );

        // Extract gig link
        const gigLink = extractGigLink();

        return {
          success: true,
          messages: messages,
          formattedArray: formattedArray,
          structuredMessages: structuredMessages,
          count: messages.length,
          customerCount: customerCount,
          sellerCount: sellerCount,
          customerName: customerName,
          gigLink: gigLink
        };
      }
    }

    console.log('⚠️ New selector found no messages, trying alternative message selectors...');

    // ===== FALLBACK: Try alternative message selector =====
    console.log('🔄 Trying alternative: //div[contains(@class,"message")]');

    // Try broader selector without the p requirement
    const altMessageXPath = "//div[contains(@class,'message')]";
    const altResult = document.evaluate(
      altMessageXPath,
      document,
      null,
      XPathResult.ORDERED_NODE_SNAPSHOT_TYPE,
      null
    );

    console.log(`Found ${altResult.snapshotLength} message elements with alternative selector`);

    if (altResult.snapshotLength > 0) {
      const messages = [];
      const structuredMessages = [];
      let customerName = null;
      let customerCount = 0;
      let sellerCount = 0;

      for (let i = 0; i < altResult.snapshotLength; i++) {
        const messageElement = altResult.snapshotItem(i);

        // Try to find sender - look for any p with sender-like classes
        let sender = 'Unknown';
        const senderSelectors = [
          "p[class*='a17q931cm']",  // Primary selector
          "p[class*='sender']",
          "p[class*='name']",
          "[data-sender-name]"
        ];

        for (const sel of senderSelectors) {
          const senderElem = messageElement.querySelector(sel);
          if (senderElem && senderElem.textContent.trim()) {
            sender = senderElem.textContent.trim();
            break;
          }
        }

        // Validate sender - skip if it's a warning/system message
        const invalidSenders = [
          'for added safety',
          'keep payments',
          'learn more',
          'fiverr',
          'system',
          'notice',
          'warning'
        ];

        const senderLower = sender.toLowerCase();
        const isInvalidSender = invalidSenders.some(invalid => senderLower.includes(invalid));

        if (isInvalidSender || sender.length > 50) {
          // This is not a real sender name, skip this element
          console.warn(`⚠️ Skipping invalid sender: "${sender.substring(0, 50)}..."`);
          continue;
        }

        // Extract message content
        let messageContent = '';
        const contentElements = messageElement.querySelectorAll('p');

        for (const elem of contentElements) {
          const elemClass = elem.className || '';
          const text = elem.textContent?.trim();

          // Skip sender name elements
          if (text && text !== sender &&
            !elemClass.includes('a17q931cm') &&
            !elemClass.includes('sender') &&
            !elemClass.includes('name')) {
            messageContent = text;
            break;
          }
        }

        // If no content found, get all text and remove sender
        if (!messageContent) {
          const allText = messageElement.textContent?.trim() || '';
          messageContent = allText.replace(sender, '').trim();
        }

        if (!messageContent || messageContent.length < 2) {
          continue;
        }

        // Remove timestamp patterns from message content
        messageContent = messageContent.replace(/^[A-Z][a-z]{2}\s+\d{1,2},\s+\d{1,2}:\d{2}\s+[AP]M/i, '').trim();
        messageContent = messageContent.replace(/^\d{1,2}\/\d{1,2},\s+\d{1,2}:\d{2}\s+[AP]M/i, '').trim();
        messageContent = messageContent.replace(/^\d{1,2}:\d{2}\s+[AP]M/i, '').trim();

        // If content is still too short after cleaning, skip
        if (messageContent.length < 2) {
          console.warn(`⚠️ Content too short after cleaning: "${messageContent}"`);
          continue;
        }

        // Determine role
        const isSellerMessage = sender.toLowerCase() === 'me' ||
          messageElement.classList.contains('seller') ||
          messageElement.classList.contains('outgoing') ||
          messageElement.querySelector('[data-sender="seller"]') ||
          determineMessageRole(messageElement, i) === 'assistant';

        const role = isSellerMessage ? 'assistant' : 'user';

        // Track customer name
        if (role === 'user' && !customerName && sender.toLowerCase() !== 'me' && sender !== 'Unknown') {
          customerName = sender;
        }

        // Count by role
        if (role === 'user') {
          customerCount++;
        } else {
          sellerCount++;
        }

        // Add to arrays
        messages.push(messageContent);
        structuredMessages.push({
          text: messageContent,
          content: messageContent,
          role: role,
          sender: sender,
          isCustomer: role === 'user',
          originalIndex: i,
          timestamp: new Date().toISOString()
        });

        console.log(`   ${i + 1}. [${role === 'user' ? '👤' : '🤝'}] ${sender}: "${messageContent.substring(0, 50)}${messageContent.length > 50 ? '...' : ''}"`);
      }

      if (messages.length > 0) {
        console.log(`✅ Successfully extracted ${messages.length} messages with fallback`);
        console.log(`   - Customer (${customerName || 'Unknown'}): ${customerCount} messages`);
        console.log(`   - Seller (Me): ${sellerCount} messages`);

        // Create formatted array
        const formattedArray = structuredMessages.map(msg =>
          `${msg.sender}: ${msg.content}`
        );

        // Extract gig link
        const gigLink = extractGigLink();

        return {
          success: true,
          messages: messages,
          formattedArray: formattedArray,
          structuredMessages: structuredMessages,
          count: messages.length,
          customerCount: customerCount,
          sellerCount: sellerCount,
          customerName: customerName,
          gigLink: gigLink
        };
      }
    }

    // Final fallback: no structured extraction possible
    console.log('❌ All message extraction methods failed');
    return {
      success: false,
      error: 'No messages found in conversation'
    };

  } catch (error) {
    console.error('Error reading messages:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Generate AI reply using OpenAI API with gig context
 */
/**
 * Estimate token count (rough approximation: ~4 chars = 1 token)
 */
function estimateTokens(text) {
  return Math.ceil(text.length / 4);
}

/**
 * Truncate messages to fit within token limit
 */
function truncateMessages(messages, maxTokens = 10000) {
  console.log(`📊 Original message count: ${messages.length}`);

  // Always keep the most recent messages (last 5)
  const recentCount = Math.min(5, messages.length);
  let selectedMessages = messages.slice(-recentCount);

  let totalText = selectedMessages.join('\n---\n');
  let estimatedTokens = estimateTokens(totalText);

  console.log(`📊 Recent ${recentCount} messages: ~${estimatedTokens} tokens`);

  // If still too large, reduce further
  if (estimatedTokens > maxTokens) {
    console.log('⚠️ Still too large, reducing to last 3 messages...');
    selectedMessages = messages.slice(-3);
    totalText = selectedMessages.join('\n---\n');
    estimatedTokens = estimateTokens(totalText);

    // If STILL too large, take only last message
    if (estimatedTokens > maxTokens) {
      console.log('⚠️ Still too large, taking only last message...');
      selectedMessages = messages.slice(-1);
      totalText = selectedMessages.join('\n---\n');
      estimatedTokens = estimateTokens(totalText);
    }
  }

  const truncated = selectedMessages.length < messages.length;

  if (truncated) {
    console.log(`✂️ Truncated from ${messages.length} to ${selectedMessages.length} messages`);
  }

  return {
    messages: selectedMessages,
    text: totalText,
    estimatedTokens: estimatedTokens,
    truncated: truncated,
    originalCount: messages.length,
    selectedCount: selectedMessages.length
  };
}

/**
 * Guess the message role (customer vs seller) using DOM hints
 */
function determineMessageRole(element, fallbackIndex = 0) {
  if (!element) {
    return fallbackIndex % 2 === 0 ? 'user' : 'assistant';
  }

  const attributeCandidates = [
    element.getAttribute('data-testid'),
    element.getAttribute('data-sender'),
    element.getAttribute('data-author'),
    element.getAttribute('aria-label'),
    element.className
  ];

  const parentTestId = element.closest('[data-testid]')?.getAttribute('data-testid');
  if (parentTestId) {
    attributeCandidates.push(parentTestId);
  }

  const attributeText = attributeCandidates
    .filter(Boolean)
    .map(attr => attr.toLowerCase())
    .join(' ');

  const buyerHints = /(buyer|customer|client|incoming|them|requester)/;
  const sellerHints = /(seller|you|outgoing|self|agent|me|by you)/;

  if (buyerHints.test(attributeText)) {
    return 'user';
  }

  if (sellerHints.test(attributeText)) {
    return 'assistant';
  }

  try {
    const style = window.getComputedStyle(element);
    const textAlign = style.textAlign || '';
    const justify = style.justifyContent || '';

    if (textAlign === 'right' || justify === 'flex-end') {
      return 'assistant';
    }

    if (textAlign === 'left' || justify === 'flex-start') {
      return 'user';
    }
  } catch (_) {
    // Ignore style lookup errors
  }

  const directionHint = element.getAttribute('dir');
  if (directionHint === 'rtl') {
    return 'assistant';
  }

  return fallbackIndex % 2 === 0 ? 'user' : 'assistant';
}

function getMessageSourceHints(element) {
  if (!element) {
    return {};
  }

  let textAlign = '';
  try {
    textAlign = window.getComputedStyle(element).textAlign || '';
  } catch (_) {
    textAlign = '';
  }

  return {
    className: element.className || '',
    dataTestId: element.getAttribute('data-testid') || '',
    ariaLabel: element.getAttribute('aria-label') || '',
    dir: element.getAttribute('dir') || '',
    textAlign
  };
}

/**
 * Get all customer messages that come after the last assistant message
 * This ensures we respond to ALL unanswered customer messages, not just the last one
 * @param {Array<string>} messages
 * @param {Array<Object>} structuredMessages
 * @returns {{ messages: Array<string>, lastMessage: string, lastIndex: number, unansweredCount: number }}
 */
function getLastCustomerMessage(messages = [], structuredMessages = null) {
  const structured = Array.isArray(structuredMessages) && structuredMessages.length > 0
    ? structuredMessages
    : null;

  const fallbackMessages = Array.isArray(messages) ? messages : [];

  if (!structured && fallbackMessages.length === 0) {
    return { messages: [], lastMessage: '', lastIndex: -1, unansweredCount: 0 };
  }

  const total = structured ? structured.length : fallbackMessages.length;

  if (!structured) {
    // Fallback: just return last message if no structured data
    for (let i = total - 1; i >= 0; i--) {
      const text = typeof fallbackMessages[i] === 'string' ? fallbackMessages[i].trim() : '';
      if (text.length > 0) {
        return {
          messages: [text],
          lastMessage: text,
          lastIndex: i,
          unansweredCount: 1
        };
      }
    }
    return { messages: [], lastMessage: '', lastIndex: -1, unansweredCount: 0 };
  }

  // Find the last assistant message index
  let lastAssistantIndex = -1;
  for (let i = total - 1; i >= 0; i--) {
    const msg = structured[i];
    if (msg.role === 'assistant') {
      lastAssistantIndex = i;
      break;
    }
  }

  console.log(`🔍 Last assistant message at index: ${lastAssistantIndex}`);

  // Collect all customer messages after the last assistant message
  const customerMessagesAfterAssistant = [];
  const startIndex = lastAssistantIndex + 1;

  for (let i = startIndex; i < total; i++) {
    const msg = structured[i];
    if (msg.role === 'user') {
      const text = (msg.text || msg.content || '').trim();
      if (text) {
        customerMessagesAfterAssistant.push({
          text: text,
          index: msg.originalIndex ?? i,
          sender: msg.sender || 'Customer'
        });
      }
    }
  }

  console.log(`📊 Found ${customerMessagesAfterAssistant.length} unanswered customer message(s) after last assistant reply`);

  if (customerMessagesAfterAssistant.length > 0) {
    // Log all unanswered messages
    customerMessagesAfterAssistant.forEach((msg, idx) => {
      console.log(`   ${idx + 1}. [${msg.sender}] "${msg.text.substring(0, 50)}${msg.text.length > 50 ? '...' : ''}"`);
    });

    // Return the last customer message (most recent) as primary, but include all
    const lastMsg = customerMessagesAfterAssistant[customerMessagesAfterAssistant.length - 1];
    return {
      messages: customerMessagesAfterAssistant.map(m => m.text),
      lastMessage: lastMsg.text,
      lastIndex: lastMsg.index,
      unansweredCount: customerMessagesAfterAssistant.length
    };
  }

  // Fallback: if no assistant message found, return the last customer message
  console.log('⚠️ No assistant message found, returning last customer message');
  for (let i = total - 1; i >= 0; i--) {
    const msg = structured[i];
    if (msg.role === 'user') {
      const text = (msg.text || msg.content || '').trim();
      if (text) {
        return {
          messages: [text],
          lastMessage: text,
          lastIndex: msg.originalIndex ?? i,
          unansweredCount: 1
        };
      }
    }
  }

  // Final fallback: return last message regardless of role
  for (let i = total - 1; i >= 0; i--) {
    const text = (structured[i]?.text || structured[i]?.content || '').trim();
    if (text.length > 0) {
      return {
        messages: [text],
        lastMessage: text,
        lastIndex: structured[i]?.originalIndex ?? i,
        unansweredCount: 1
      };
    }
  }

  return { messages: [], lastMessage: '', lastIndex: -1, unansweredCount: 0 };
}

async function generateAIReply(messages, apiKey, gigInfo = null) {
  console.log('🤖 Generating AI reply via backend...');

  try {
    // Format messages for backend API
    // Convert messages array to the format expected by backend
    const formattedMessages = messages.map((msg, index) => {
      // Determine role based on message position (odd=buyer, even=seller for most Fiverr convos)
      const role = index % 2 === 0 ? 'buyer' : 'seller';
      return {
        role: role,
        content: msg
      };
    });

    // Get buyer username from current page if available
    let buyerUsername = null;
    try {
      const usernameElement = document.querySelector('[data-username]');
      buyerUsername = usernameElement ? usernameElement.getAttribute('data-username') : null;
    } catch (e) {
      // Ignore if element not found
    }

    // Get conversation ID from URL if available
    let conversationId = null;
    try {
      const urlMatch = window.location.pathname.match(/\/inbox\/([^\/]+)/);
      conversationId = urlMatch ? urlMatch[1] : null;
    } catch (e) {
      // Ignore
    }

    console.log(gigInfo && gigInfo.matched ? '✅ Using gig-specific context' : '⚠️ Using generic context');

    // Call backend API (apiKey parameter is ignored, backend uses its own OpenAI key)
    const result = await api.generateAIReply(formattedMessages, gigInfo, buyerUsername, conversationId);

    if (!result.success) {
      return {
        success: false,
        error: result.message || 'Failed to generate reply'
      };
    }

    console.log('✅ AI reply generated via backend');

    return {
      success: true,
      reply: result.reply,
      truncated: false, // Backend handles truncation
      messageCount: formattedMessages.length,
      totalMessages: messages.length
    };

  } catch (error) {
    console.error('Error generating AI reply:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Type reply in textarea (human-like)
 */
async function typeReply(replyText) {
  console.log('Typing reply in textarea...');

  try {
    const textareaXPath = "//textarea[@id='send-message-text-area']";

    const textarea = document.evaluate(
      textareaXPath,
      document,
      null,
      XPathResult.FIRST_ORDERED_NODE_TYPE,
      null
    ).singleNodeValue;

    if (!textarea) {
      return {
        success: false,
        error: 'Message textarea not found on page'
      };
    }

    textarea.value = '';
    textarea.focus();

    for (let i = 0; i < replyText.length; i++) {
      textarea.value += replyText[i];

      const inputEvent = new Event('input', { bubbles: true });
      textarea.dispatchEvent(inputEvent);

      const delay = Math.random() * 50 + 30;
      await sleep(delay);

      if (Math.random() < 0.05) {
        await sleep(Math.random() * 200 + 100);
      }
    }

    const changeEvent = new Event('change', { bubbles: true });
    textarea.dispatchEvent(changeEvent);

    console.log('✅ Reply typed successfully');

    return {
      success: true,
      message: 'Reply typed successfully'
    };

  } catch (error) {
    console.error('Error typing reply:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Click send button after typing
 */
async function clickSendButton() {
  console.log('Clicking send button...');

  try {
    const sendButtonXPath = "//button[@aria-label='Send']";

    const sendButton = document.evaluate(
      sendButtonXPath,
      document,
      null,
      XPathResult.FIRST_ORDERED_NODE_TYPE,
      null
    ).singleNodeValue;

    if (!sendButton) {
      return {
        success: false,
        error: 'Send button not found on page'
      };
    }

    console.log('Send button found, clicking...');
    sendButton.click();

    // Wait for message to be sent
    await sleep(2000);

    console.log('✅ Send button clicked successfully');

    return {
      success: true,
      message: 'Send button clicked successfully'
    };

  } catch (error) {
    console.error('Error clicking send button:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Check if there are more unread messages on inbox page
 * NEW WORKFLOW: Click DIV → Click P (Unread) → Check unreads → If none, Click DIV → Click "All messages" → Repeat cycle
 */
async function checkMoreUnreads() {
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('🚀 checkMoreUnreads() FUNCTION CALLED');
  console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
  console.log('Checking for more unread messages...');

  try {
    // Wait a bit for page to update
    await sleep(1000);

    const unreadSelectors = [
      "//div[@class='_1v0154n0 contact first'][.//div[@class='_1hjuhhy14 _1hjuhhy19 _1hjuhhy1j qem7dds _1v0154n0 _1v0154n10j _1v0154n1j _1v0154n73 _1v0154nzu _1v0154n17c _1v0154n195 _1v0154n25m _1v0154n1b8']]",
      "//div[@class='_1v0154n0 contact'][.//div[@class='_1hjuhhy14 _1hjuhhy19 _1hjuhhy1j qem7dds _1v0154n0 _1v0154n10j _1v0154n1j _1v0154n73 _1v0154nzu _1v0154n17c _1v0154n195 _1v0154n25m _1v0154n1b8']]",
      "//div[contains(@class, 'contact')][.//p[contains(@class, 'a17q931eu') and normalize-space(text()) != '']]"
    ];

    // First, check if there are already unread messages visible
    for (const selector of unreadSelectors) {
      try {
        const unreadContact = document.evaluate(
          selector,
          document,
          null,
          XPathResult.FIRST_ORDERED_NODE_TYPE,
          null
        ).singleNodeValue;

        if (unreadContact) {
          console.log('✅ More unread messages found!');
          return {
            success: true,
            hasMore: true
          };
        }
      } catch (error) {
        continue;
      }
    }

    console.log('❌ No more unread messages found initially');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

    // ✨ Get refresh mode state from background
    console.log('📡 Checking refresh mode state...');
    const refreshModeEnabled = await new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'getState' }, (response) => {
        resolve(response?.state?.refreshMode || false);
      });
    });

    console.log(`📊 Refresh Mode: ${refreshModeEnabled ? 'ON ✅' : 'OFF ❌'}`);
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

    // ✨ NEW WORKFLOW: If refresh mode is OFF, execute the dual-pass workflow
    if (!refreshModeEnabled) {
      console.log('🔄 Refresh mode is OFF - Starting DUAL-PASS message check workflow...');
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

      // Helper function to click DIV and P elements
      const clickDivAndP = async (pText) => {
        console.log('');
        console.log(`📍 Searching for DIV element`);
        console.log('   XPath: //div[@class="a17q930 a17q9310e a17q93195 a17q931b8"]');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        const divXPath = "//div[@class='a17q930 a17q9310e a17q93195 a17q931b8']";
        let divElement = null;
        let divAttempts = 0;
        const maxDivAttempts = 10;

        while (!divElement && divAttempts < maxDivAttempts) {
          divAttempts++;
          console.log(`🔍 [DIV] Attempt ${divAttempts}/${maxDivAttempts} - Searching...`);

          divElement = document.evaluate(
            divXPath,
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
          ).singleNodeValue;

          if (divElement) {
            console.log(`✅ [DIV] Found on attempt ${divAttempts}!`);
          } else {
            console.log(`❌ [DIV] Not found. Waiting 2 seconds...`);
            await sleep(2000);
          }
        }

        if (!divElement) {
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('⚠️ [DIV] Element not found even after 10 attempts');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          return false;
        }

        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log('🖱️ [DIV] Clicking element...');
        divElement.click();
        console.log('✅ [DIV] Clicked successfully!');
        console.log('⏳ Waiting 1.5 seconds...');
        await sleep(1500);
        console.log('');

        // Now click P element
        console.log(`📍 Searching for P element: "${pText}"`);
        console.log(`   XPath: //p[normalize-space()='${pText}']`);
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        const pXPath = `//p[normalize-space()='${pText}']`;
        let pElement = null;
        let pAttempts = 0;
        const maxPAttempts = 10;

        while (!pElement && pAttempts < maxPAttempts) {
          pAttempts++;
          console.log(`🔍 [P] Attempt ${pAttempts}/${maxPAttempts} - Searching...`);

          pElement = document.evaluate(
            pXPath,
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
          ).singleNodeValue;

          if (pElement) {
            console.log(`✅ [P] Found on attempt ${pAttempts}!`);
          } else {
            console.log(`❌ [P] Not found. Waiting 2 seconds...`);
            await sleep(2000);
          }
        }

        if (!pElement) {
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('⚠️ [P] Element not found even after 10 attempts');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          return false;
        }

        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log('🖱️ [P] Clicking element...');
        pElement.click();
        console.log('✅ [P] Clicked successfully!');
        console.log('⏳ Waiting 3 seconds for page refresh...');
        await sleep(3000);
        console.log('✅ Wait complete!');
        console.log('');

        return true;
      };

      // Helper function to check for unread messages
      const checkForUnreads = async () => {
        console.log('📍 Checking for unread messages...');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        let selectorIndex = 0;
        for (const selector of unreadSelectors) {
          try {
            selectorIndex++;
            console.log(`🔍 Trying selector ${selectorIndex}/${unreadSelectors.length}...`);

            const unreadContact = document.evaluate(
              selector,
              document,
              null,
              XPathResult.FIRST_ORDERED_NODE_TYPE,
              null
            ).singleNodeValue;

            if (unreadContact) {
              console.log(`✅ Unread message found with selector ${selectorIndex}!`);
              return true;
            } else {
              console.log(`❌ Selector ${selectorIndex} found no unread messages`);
            }
          } catch (error) {
            console.log(`⚠️ Selector ${selectorIndex} threw error: ${error.message}`);
            continue;
          }
        }

        console.log('❌ No unread messages found');
        return false;
      };

      try {
        // ═══════════════════════════════════════════════════════════════
        // CYCLE 1: Click DIV → Click "Unread" → Check for unreads
        // ═══════════════════════════════════════════════════════════════
        console.log('');
        console.log('╔═══════════════════════════════════════════════════════╗');
        console.log('║           CYCLE 1: UNREAD MESSAGES CHECK              ║');
        console.log('╚═══════════════════════════════════════════════════════╝');
        console.log('');

        let cycle1Success = await clickDivAndP('Unread');

        if (cycle1Success) {
          let foundUnread = await checkForUnreads();

          if (foundUnread) {
            console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            console.log('🎉 CYCLE 1 SUCCESS: Unread messages found!');
            console.log('   Bot will now reply to all unread messages using previous rules');
            console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            return {
              success: true,
              hasMore: true,
              refreshed: true,
              cycle: 1,
              messageType: 'unread'
            };
          }

          // No unread messages found, proceed to "All messages"
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('ℹ️ CYCLE 1: No unread messages found');
          console.log('   Proceeding to check "All messages"...');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('');

          // Click DIV → Click "All messages"
          console.log('╔═══════════════════════════════════════════════════════╗');
          console.log('║        CYCLE 1: ALL MESSAGES CHECK (FALLBACK)         ║');
          console.log('╚═══════════════════════════════════════════════════════╝');
          console.log('');

          let allMessagesSuccess = await clickDivAndP('All messages');

          if (allMessagesSuccess) {
            foundUnread = await checkForUnreads();

            if (foundUnread) {
              console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
              console.log('🎉 CYCLE 1 SUCCESS: Messages found in "All messages"!');
              console.log('   Bot will now reply to all messages using previous rules');
              console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
              return {
                success: true,
                hasMore: true,
                refreshed: true,
                cycle: 1,
                messageType: 'all'
              };
            }
          }
        }

        // ═══════════════════════════════════════════════════════════════
        // CYCLE 2: Repeat the entire process
        // ═══════════════════════════════════════════════════════════════
        console.log('');
        console.log('╔═══════════════════════════════════════════════════════╗');
        console.log('║           CYCLE 2: UNREAD MESSAGES CHECK              ║');
        console.log('╚═══════════════════════════════════════════════════════╝');
        console.log('');

        let cycle2Success = await clickDivAndP('Unread');

        if (cycle2Success) {
          let foundUnread = await checkForUnreads();

          if (foundUnread) {
            console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            console.log('🎉 CYCLE 2 SUCCESS: Unread messages found!');
            console.log('   Bot will now reply to all unread messages using previous rules');
            console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            return {
              success: true,
              hasMore: true,
              refreshed: true,
              cycle: 2,
              messageType: 'unread'
            };
          }

          // No unread messages found, proceed to "All messages"
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('ℹ️ CYCLE 2: No unread messages found');
          console.log('   Proceeding to check "All messages"...');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('');

          // Click DIV → Click "All messages"
          console.log('╔═══════════════════════════════════════════════════════╗');
          console.log('║        CYCLE 2: ALL MESSAGES CHECK (FINAL)            ║');
          console.log('╚═══════════════════════════════════════════════════════╝');
          console.log('');

          let allMessagesSuccess = await clickDivAndP('All messages');

          if (allMessagesSuccess) {
            foundUnread = await checkForUnreads();

            if (foundUnread) {
              console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
              console.log('🎉 CYCLE 2 SUCCESS: Messages found in "All messages"!');
              console.log('   Bot will now reply to all messages using previous rules');
              console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
              return {
                success: true,
                hasMore: true,
                refreshed: true,
                cycle: 2,
                messageType: 'all'
              };
            }
          }
        }

        // ═══════════════════════════════════════════════════════════════
        // Both cycles complete - no messages found
        // ═══════════════════════════════════════════════════════════════
        console.log('');
        console.log('╔═══════════════════════════════════════════════════════╗');
        console.log('║              DUAL-PASS WORKFLOW COMPLETE              ║');
        console.log('╚═══════════════════════════════════════════════════════╝');
        console.log('');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log('ℹ️ No messages found after 2 complete cycles');
        console.log('   Checked: Unread → All messages (Cycle 1)');
        console.log('   Checked: Unread → All messages (Cycle 2)');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        return {
          success: true,
          hasMore: false,
          refreshed: true,
          cyclesCompleted: 2
        };

      } catch (refreshError) {
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.error('❌ ERROR during dual-pass workflow:');
        console.error('   Message:', refreshError.message);
        console.error('   Stack:', refreshError.stack);
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      }
    } else {
      console.log('ℹ️ Refresh mode is ON - Dual-pass workflow SKIPPED');
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    }

    return {
      success: true,
      hasMore: false
    };

  } catch (error) {
    console.error('Error checking for more unreads:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Complete read and reply flow with send button click
 */
async function readAndReply(apiKey, userGigs = [], userId = null) {
  // ⏱️ Start timer - measure time from unread click to send button click
  const startTime = Date.now();
  console.log('⏱️ Response timer started');

  console.log('=== Starting Read and Reply Flow ===');
  console.log(`📦 Received ${userGigs.length} gig(s) for matching`);

  // Extract user ID from URL if not provided
  if (!userId) {
    userId = extractUserIdFromURL();
  }

  console.log(`👤 User ID: ${userId || 'Unknown (will use timestamp)'}`);

  try {
    await sleep(2000);

    // Step 0: Scroll to bottom to load all lazy content, then scroll to top
    console.log('Step 0: Starting scroll sequence to load all content...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '📜 Scrolling to load all conversation history...'
    }).catch(() => { });

    // Find the content container (message area)
    const contentContainer = document.querySelector('div.content');

    if (contentContainer) {
      console.log('✅ Found content container');
      console.log(`📏 Initial scrollTop: ${contentContainer.scrollTop}px`);
      console.log(`📏 Initial scrollHeight: ${contentContainer.scrollHeight}px`);
      console.log('');

      // Native smooth scroll to top
      console.log('=== Native Smooth Scroll to TOP ===');

      const targetXPath = "//p[@class='_1g8w6ydk a17q931ez a17q931ch a17q938 a17q932']";

      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: '⬆️ Smooth scrolling to load messages...'
      }).catch(() => { });

      let targetFound = false;
      let checkCount = 0;
      const maxChecks = 100;

      console.log(`🔍 Target: ${targetXPath}`);
      console.log('🎯 UNIQUE APPROACH: Continuous scroll with DOM monitoring');
      console.log('');

      // Step 1: Scroll to top gradually while checking
      console.log('📜 Phase 1: Scrolling to TOP...');
      let currentHeight = contentContainer.scrollHeight;

      while (contentContainer.scrollTop > 100 && checkCount < maxChecks) {
        checkCount++;

        // Scroll up in chunks
        contentContainer.scrollTop = Math.max(0, contentContainer.scrollTop - 800);
        await new Promise(resolve => setTimeout(resolve, 200));

        // Check target every 5 scrolls
        if (checkCount % 5 === 0) {
          const check = document.evaluate(targetXPath, document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
          if (check) {
            console.log(`   ✅ TARGET FOUND during scroll at check ${checkCount}!`);
            targetFound = true;
            break;
          }
          console.log(`   📍 Check ${checkCount}: ${contentContainer.scrollTop}px`);
        }
      }

      if (targetFound) {
        chrome.runtime.sendMessage({ action: 'addLog', type: 'SUCCESS', message: `✅ Found target after ${checkCount} checks!` }).catch(() => { });
      } else {
        // Step 2: At top - aggressive lazy loading
        console.log('');
        console.log('📜 Phase 2: AT TOP - Aggressive lazy loading...');
        contentContainer.scrollTop = 0;
        await new Promise(resolve => setTimeout(resolve, 500));

        for (let cycle = 1; cycle <= 50 && !targetFound; cycle++) {
          // Big movements to trigger lazy load
          contentContainer.scrollTop = 0;
          await new Promise(resolve => setTimeout(resolve, 400));

          contentContainer.scrollTop = 600;
          await new Promise(resolve => setTimeout(resolve, 400));

          contentContainer.scrollTop = 0;
          await new Promise(resolve => setTimeout(resolve, 400));

          // Check if height increased (new content loaded)
          const newHeight = contentContainer.scrollHeight;
          if (newHeight > currentHeight) {
            console.log(`   📈 Cycle ${cycle}: Content loaded! ${currentHeight}px → ${newHeight}px`);
            currentHeight = newHeight;
          }

          // Check for target
          const check = document.evaluate(targetXPath, document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
          if (check) {
            console.log(`   ✅ TARGET FOUND at cycle ${cycle}!`);
            targetFound = true;
            chrome.runtime.sendMessage({ action: 'addLog', type: 'SUCCESS', message: `✅ Found target after ${cycle} cycles!` }).catch(() => { });
            break;
          }

          if (cycle % 10 === 0) {
            console.log(`   🔄 Cycle ${cycle}/50: Still loading... (Height: ${newHeight}px)`);
          }

          // If no new content for 15 cycles, break
          if (cycle > 15 && newHeight === currentHeight) {
            console.log(`   ⚠️ No new content after ${cycle} cycles. Proceeding...`);
            break;
          }
        }
      }

      // Final position
      contentContainer.scrollTop = 0;
      console.log(`\n📊 Total checks: ${checkCount}`);
      console.log(`🎯 Target found: ${targetFound}`);
      console.log('');

      console.log(`📊 Total checks: ${checkCount}`);
      console.log(`🎯 Target found: ${targetFound}`);
      console.log('');

      chrome.runtime.sendMessage({
        action: 'addLog',
        type: targetFound ? 'SUCCESS' : 'WARNING',
        message: targetFound
          ? `✅ Ready! Found target in ${checkCount} checks`
          : `⚠️ Target not found after ${checkCount} checks`
      }).catch(() => { });

      // Extra wait for DOM to settle
      console.log('⏳ Waiting for DOM to settle...');
      await sleep(1000);

    } else {
      console.log('⚠️ Content container not found, using window scroll...');
      console.log('');
      console.log('=== Native Smooth Window Scroll ===');

      const targetXPath = "//p[@class='_1g8w6ydk a17q931ez a17q931ch a17q938 a17q932']";

      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: '⬆️ Smooth window scroll...'
      }).catch(() => { });

      let targetFound = false;
      let checkCount = 0;
      const maxChecks = 100;

      console.log(`🔍 Target: ${targetXPath}`);
      console.log('🎯 UNIQUE APPROACH: Continuous scroll with DOM monitoring');
      console.log('');

      // Step 1: Scroll to top gradually while checking
      console.log('📜 Phase 1: Scrolling to TOP...');
      let currentHeight = Math.max(document.body.scrollHeight, document.documentElement.scrollHeight);

      while (window.scrollY > 100 && checkCount < maxChecks) {
        checkCount++;

        // Scroll up in chunks
        window.scrollTo(0, Math.max(0, window.scrollY - 800));
        await new Promise(resolve => setTimeout(resolve, 200));

        // Check target every 5 scrolls
        if (checkCount % 5 === 0) {
          const check = document.evaluate(targetXPath, document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
          if (check) {
            console.log(`   ✅ TARGET FOUND during scroll at check ${checkCount}!`);
            targetFound = true;
            break;
          }
          console.log(`   📍 Check ${checkCount}: ${window.scrollY}px`);
        }
      }

      if (targetFound) {
        chrome.runtime.sendMessage({ action: 'addLog', type: 'SUCCESS', message: `✅ Found target after ${checkCount} checks!` }).catch(() => { });
      } else {
        // Step 2: At top - aggressive lazy loading
        console.log('');
        console.log('📜 Phase 2: AT TOP - Aggressive lazy loading...');
        window.scrollTo(0, 0);
        await new Promise(resolve => setTimeout(resolve, 500));

        for (let cycle = 1; cycle <= 50 && !targetFound; cycle++) {
          // Big movements to trigger lazy load
          window.scrollTo(0, 0);
          await new Promise(resolve => setTimeout(resolve, 400));

          window.scrollTo(0, 600);
          await new Promise(resolve => setTimeout(resolve, 400));

          window.scrollTo(0, 0);
          await new Promise(resolve => setTimeout(resolve, 400));

          // Check if height increased
          const newHeight = Math.max(document.body.scrollHeight, document.documentElement.scrollHeight);
          if (newHeight > currentHeight) {
            console.log(`   📈 Cycle ${cycle}: Content loaded! ${currentHeight}px → ${newHeight}px`);
            currentHeight = newHeight;
          }

          // Check for target
          const check = document.evaluate(targetXPath, document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
          if (check) {
            console.log(`   ✅ TARGET FOUND at cycle ${cycle}!`);
            targetFound = true;
            chrome.runtime.sendMessage({ action: 'addLog', type: 'SUCCESS', message: `✅ Found target after ${cycle} cycles!` }).catch(() => { });
            break;
          }

          if (cycle % 10 === 0) {
            console.log(`   🔄 Cycle ${cycle}/50: Still loading... (Height: ${newHeight}px)`);
          }

          // If no new content for 15 cycles, break
          if (cycle > 15 && newHeight === currentHeight) {
            console.log(`   ⚠️ No new content after ${cycle} cycles. Proceeding...`);
            break;
          }
        }
      }

      // Final position
      window.scrollTo(0, 0);
      console.log(`\n📊 Total checks: ${checkCount}`);
      console.log(`🎯 Target found: ${targetFound}`);
      console.log('');

      console.log(`📊 Total checks: ${checkCount}`);
      console.log(`🎯 Target found: ${targetFound}`);
      console.log('');

      chrome.runtime.sendMessage({
        action: 'addLog',
        type: targetFound ? 'SUCCESS' : 'WARNING',
        message: targetFound
          ? `✅ Ready! Found target in ${checkCount} checks`
          : `⚠️ Target not found after ${checkCount} checks`
      }).catch(() => { });

      console.log('⏳ Waiting for DOM to settle...');
      await sleep(1000);
    }

    // Extra wait to ensure all lazy loaded content is ready
    console.log('⏳ Waiting for all content to fully load...');
    await sleep(4000); // Increased from 3000ms to 4000ms for content to settle

    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '⏳ Waiting for content to fully load...'
    }).catch(() => { });

    // Step 1: Read messages
    console.log('Step 1: Reading messages...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '📖 Reading messages and extracting gig link...'
    }).catch(() => { });

    const readResult = await readMessageContent();

    if (!readResult.success) {
      return {
        success: false,
        error: `Failed to read messages: ${readResult.error}`,
        step: 'read_messages'
      };
    }

    console.log('');
    console.log('=== 📊 Conversation Extraction Summary ===');
    console.log(`✅ Total messages: ${readResult.count}`);
    console.log(`   👤 Customer (${readResult.customerName || 'Unknown'}): ${readResult.customerCount || 'Unknown'} messages`);
    console.log(`   🤝 Seller (Me): ${readResult.sellerCount || 'Unknown'} messages`);
    console.log(`🔗 Gig link: ${readResult.gigLink || 'Not found'}`);
    console.log('===========================================');
    console.log('');

    // Show formatted array if available
    if (readResult.formattedArray && readResult.formattedArray.length > 0) {
      console.log('📋 Full Conversation (Clean Single Array):');
      readResult.formattedArray.forEach((line, i) => {
        console.log(`   ${i + 1}. ${line}`);
      });
      console.log('');
    }

    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'SUCCESS',
      message: `📖 Read ${readResult.count} message(s) from ${readResult.customerName || 'customer'} (👤${readResult.customerCount || '?'} customer, 🤝${readResult.sellerCount || '?'} seller)`
    }).catch(() => { });

    if (readResult.gigLink) {
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: `🔗 Gig link found: ${readResult.gigLink}`
      }).catch(() => { });
    } else {
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'WARNING',
        message: '⚠️ No gig link found in conversation'
      }).catch(() => { });
    }

    // Wait a bit before processing gig matching
    await sleep(1000);

    // Step 2: Match gig if link found
    let gigInfo = null;

    console.log('');
    console.log('=== Step 2: Gig Matching Process ===');
    console.log(`📦 Checking ${userGigs.length} gig(s) from database...`);

    if (readResult.gigLink && userGigs.length > 0) {
      console.log('✅ Prerequisites met:');
      console.log(`   - Gig link found: ${readResult.gigLink}`);
      console.log(`   - User has ${userGigs.length} gig(s) in database`);
      console.log('');
      console.log('🔍 Starting gig matching...');

      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: `🔍 Matching gig link with ${userGigs.length} database gig(s)...`
      }).catch(() => { });

      // Log all database gigs for debugging
      console.log('📋 Database gigs received:');
      userGigs.forEach((gig, index) => {
        console.log(`   Gig #${index + 1}:`);
        console.log(`      Title: ${gig.title}`);
        console.log(`      Link: ${gig.link}`);
      });
      console.log('');

      // Normalize gig link (extract path from URL and ensure consistent format)
      const normalizeLink = (link) => {
        if (!link) return '';

        // Remove whitespace
        link = link.trim();

        try {
          // If it's a full URL (https://www.fiverr.com/username/gig-slug)
          if (link.startsWith('http')) {
            const url = new URL(link);
            // Extract pathname: /username/gig-slug
            let path = url.pathname;

            // Remove trailing slash if present
            if (path.endsWith('/')) {
              path = path.slice(0, -1);
            }

            return path;
          }

          // If it's already a path (/username/gig-slug)
          if (link.startsWith('/')) {
            // Remove trailing slash if present
            return link.endsWith('/') ? link.slice(0, -1) : link;
          }

          // If it's just username/gig-slug (no leading /)
          return '/' + link;

        } catch (error) {
          console.warn('Error normalizing link:', link, error);
          // Fallback: ensure it starts with /
          return link.startsWith('/') ? link : `/${link}`;
        }
      };

      const extractedPath = normalizeLink(readResult.gigLink);
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('📍 EXTRACTED FROM PAGE:');
      console.log(`   Raw: "${readResult.gigLink}"`);
      console.log(`   Normalized: "${extractedPath}"`);
      console.log(`   Length: ${extractedPath.length} chars`);
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('');
      console.log('🔄 COMPARING WITH DATABASE GIGS:');
      console.log('');

      // Try to find matching gig
      let gigNumber = 0;
      const matchedGig = userGigs.find((gig, index) => {
        gigNumber = index + 1;
        const gigPath = normalizeLink(gig.link);

        console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
        console.log(`📦 Gig #${gigNumber}: ${gig.title}`);
        console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
        console.log(`   Database raw: "${gig.link}"`);
        console.log(`   Database normalized: "${gigPath}"`);
        console.log(`   Database length: ${gigPath.length} chars`);
        console.log('');
        console.log(`   Extracted normalized: "${extractedPath}"`);
        console.log(`   Extracted length: ${extractedPath.length} chars`);
        console.log('');
        console.log('   🔍 COMPARISON CHECKS:');

        // Try exact match first (most reliable)
        const exactMatch = gigPath === extractedPath;
        console.log(`      1. Exact match: ${exactMatch ? '✅ YES' : '❌ NO'}`);
        if (exactMatch) {
          console.log(`      ✅ MATCH TYPE: EXACT MATCH`);
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('');
          return true;
        }

        // Try lowercase comparison (case-insensitive)
        const caseInsensitiveMatch = gigPath.toLowerCase() === extractedPath.toLowerCase();
        console.log(`      2. Case-insensitive: ${caseInsensitiveMatch ? '✅ YES' : '❌ NO'}`);
        if (caseInsensitiveMatch) {
          console.log(`      ✅ MATCH TYPE: CASE-INSENSITIVE MATCH`);
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('');
          return true;
        }

        // Try partial match (one contains the other)
        const dbContainsExtracted = gigPath.includes(extractedPath);
        const extractedContainsDb = extractedPath.includes(gigPath);
        console.log(`      3. DB contains extracted: ${dbContainsExtracted ? '✅ YES' : '❌ NO'}`);
        console.log(`      4. Extracted contains DB: ${extractedContainsDb ? '✅ YES' : '❌ NO'}`);

        if (dbContainsExtracted || extractedContainsDb) {
          console.log(`      ✅ MATCH TYPE: PARTIAL MATCH`);
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
          console.log('');
          return true;
        }

        // Show character differences if no match
        console.log('');
        console.log('      ❌ NO MATCH - Character by character comparison:');
        const maxLen = Math.max(gigPath.length, extractedPath.length);
        let firstDiff = -1;
        for (let i = 0; i < maxLen; i++) {
          const dbChar = gigPath[i] || '(end)';
          const exChar = extractedPath[i] || '(end)';
          if (dbChar !== exChar) {
            firstDiff = i;
            console.log(`         Position ${i}: DB="${dbChar}" vs Extracted="${exChar}" ❌`);
            if (firstDiff > 5) break; // Show max 5 differences
          }
        }
        if (firstDiff === -1) {
          console.log('         (Identical strings - should have matched!)');
        }

        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log('');
        return false;
      });

      if (matchedGig) {
        console.log(`✅ MATCH FOUND! Gig #${gigNumber}`);
        console.log(`   Title: ${matchedGig.title}`);
        console.log(`   Has AI Summary: ${!!matchedGig.summary}`);
        console.log(`   Summary Length: ${matchedGig.summary ? matchedGig.summary.length : 0} chars`);

        if (matchedGig.summary) {
          console.log(`   Summary Preview: "${matchedGig.summary.substring(0, 100)}..."`);
        } else {
          console.log(`   ⚠️ WARNING: No AI summary found! Will use fallback context.`);
          console.log(`   💡 TIP: Save gig again to generate AI summary.`);
        }
        console.log('');

        chrome.runtime.sendMessage({
          action: 'addLog',
          type: 'SUCCESS',
          message: `✅ Gig matched! Using "${matchedGig.title}" ${matchedGig.summary ? 'with AI summary' : '(no summary)'}`
        }).catch(() => { });

        // ✅ Send ONLY AI-generated summary (not raw data)
        // Backend will use stored summary for efficient token usage
        gigInfo = {
          matched: true,
          gigNumber: gigNumber,
          title: matchedGig.title,
          summary: matchedGig.summary || ''  // ✅ Pre-generated AI summary from database
        };
      } else {
        console.log('❌ NO MATCH FOUND');
        console.log('   The extracted gig link does not match any gig in your database');
        console.log('   Possible reasons:');
        console.log('   - Gig link in conversation is different from saved gigs');
        console.log('   - Gig not added to database yet');
        console.log('   - Link format mismatch');
        console.log('');
        console.log('💡 Will use generic context for reply');

        chrome.runtime.sendMessage({
          action: 'addLog',
          type: 'WARNING',
          message: '⚠️ No matching gig found, using generic context'
        }).catch(() => { });
      }
    } else if (!readResult.gigLink) {
      console.log('⚠️ No gig link detected on conversation page');
      console.log('   This conversation may not be about a specific gig');
      console.log('   Will use generic context for reply');
    } else if (userGigs.length === 0) {
      console.log('⚠️ No gigs found in database');
      console.log('   Please add gigs in the extension to enable gig-specific replies');
      console.log('   Will use generic context for reply');
    }

    console.log('=== End of Gig Matching Process ===');
    console.log('');

    // Step 3: Generate AI reply (with conversation memory)
    console.log('Step 3: Generating AI reply with conversation memory...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '🤖 Generating AI reply with memory...'
    }).catch(() => { });

    // Get user ID (use URL or conversation context)
    const conversationUserId = userId || extractUserIdFromURL() || `user_${Date.now()}`;

    // Determine all unanswered customer messages (after last assistant reply)
    console.log('🔍 Identifying unanswered customer messages...');
    const {
      messages: unansweredMessages,
      lastMessage: lastCustomerMessage,
      lastIndex: lastCustomerIndex,
      unansweredCount
    } = getLastCustomerMessage(
      readResult.messages,
      readResult.structuredMessages
    );

    if (!lastCustomerMessage) {
      console.error('❌ No customer message detected to reply to');
      return {
        success: false,
        error: 'No customer message detected to reply to',
        step: 'generate_reply'
      };
    }

    console.log('');
    console.log('=== 🎯 Unanswered Customer Messages ===');
    console.log(`Total unanswered: ${unansweredCount} message(s)`);
    console.log(`Position: After last assistant reply`);

    if (unansweredCount > 1) {
      console.log(`\n📝 All ${unansweredCount} unanswered messages:`);
      unansweredMessages.forEach((msg, idx) => {
        console.log(`   ${idx + 1}. "${msg.substring(0, 80)}${msg.length > 80 ? '...' : ''}"`);
      });
      console.log(`\n🎯 Will respond to the most recent: "${lastCustomerMessage.substring(0, 80)}${lastCustomerMessage.length > 80 ? '...' : ''}"`);
    } else {
      console.log(`Last message: "${lastCustomerMessage.substring(0, 100)}${lastCustomerMessage.length > 100 ? '...' : ''}"`);
    }

    if (lastCustomerIndex !== readResult.messages.length - 1) {
      console.log('⚠️ Note: Latest message in DOM appears to be from seller');
      console.log('   Responding to most recent customer message instead');
    } else {
      console.log('✅ This is the most recent message in the conversation');
    }
    console.log('======================================');
    console.log('');

    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: `🎯 Found ${unansweredCount} unanswered message(s) - replying to: "${lastCustomerMessage.substring(0, 40)}..."`
    }).catch(() => { });

    // ✅ NEW BACKEND-FIRST APPROACH - Call new full backend endpoint
    console.log('🚀 Using NEW BACKEND-FIRST approach...');
    console.log(`📤 Sending to backend: customerId="${conversationUserId}", message="${lastCustomerMessage.substring(0, 50)}..."`);

    // Extract customer name from readResult
    const buyerUsername = readResult.customerName || conversationUserId;
    console.log(`👤 Buyer username: ${buyerUsername}`);

    // Log gig info being sent
    if (gigInfo) {
      console.log(`🎯 Gig Info being sent to backend:`);
      console.log(`   - Matched: ${gigInfo.matched}`);
      console.log(`   - Title: ${gigInfo.title}`);
      console.log(`   - Gig Number: ${gigInfo.gigNumber}`);
      console.log(`   - Has Summary: ${!!gigInfo.summary}`);
      console.log(`   - Summary Length: ${gigInfo.summary?.length || 0} chars`);

      if (gigInfo.summary) {
        console.log(`   ✅ Sending AI-generated summary for efficient token usage`);
      } else {
        console.log(`   ⚠️ No summary available - backend will use fallback context`);
      }
    } else {
      console.log(`⚠️ No gig info - using generic context`);
    }

    // ✅ FIXED: Don't send sellerName from extension - let backend use database value
    // This prevents buyer name from being confused as seller name
    const aiResult = await new Promise((resolve) => {
      chrome.runtime.sendMessage({
        action: 'generateReplyFull',
        customerId: conversationUserId,
        newMessage: lastCustomerMessage,
        buyerUsername: buyerUsername,
        // sellerName is intentionally NOT sent - backend will use user.sellerName from database
        gigInfo: gigInfo
      }, (response) => {
        if (chrome.runtime.lastError) {
          console.error('❌ Backend call error:', chrome.runtime.lastError);
          resolve({
            success: false,
            error: chrome.runtime.lastError.message
          });
        } else {
          resolve(response || { success: false, error: 'No response from backend' });
        }
      });
    });

    if (!aiResult.success) {
      console.error('❌ Backend reply failed:', aiResult.error);
      return {
        success: false,
        error: `Failed to generate reply: ${aiResult.error}`,
        step: 'generate_reply'
      };
    }

    console.log('✅ AI reply generated');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'SUCCESS',
      message: '✅ AI reply generated with memory'
    }).catch(() => { });

    // Log conversation memory info
    if (aiResult.message_count) {
      console.log(`💬 Conversation: ${aiResult.message_count} total messages stored`);
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: `💬 ${aiResult.message_count} messages in conversation history`
      }).catch(() => { });
    }

    if (aiResult.has_summary) {
      console.log(`🧠 Conversation summary available`);
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'INFO',
        message: `🧠 Using conversation summary for context`
      }).catch(() => { });
    }

    // Step 3.5: Scroll down to bottom before typing reply
    console.log('Step 3.5: Scrolling to bottom for reply...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '⬇️ Scrolling to bottom to type reply...'
    }).catch(() => { });

    // Find content container and scroll to bottom
    const replyContentContainer = document.querySelector('div.content');

    if (replyContentContainer) {
      console.log('✅ Found content container, scrolling to bottom...');
      console.log(`📏 Current scrollTop: ${replyContentContainer.scrollTop}px`);
      console.log(`📏 ScrollHeight: ${replyContentContainer.scrollHeight}px`);

      // Smooth scroll to bottom
      replyContentContainer.scrollTo({
        top: replyContentContainer.scrollHeight,
        behavior: 'smooth'
      });

      // Wait for scroll animation to complete
      await sleep(1500);

      console.log(`✅ Scrolled to bottom (${replyContentContainer.scrollTop}px)`);
      chrome.runtime.sendMessage({
        action: 'addLog',
        type: 'SUCCESS',
        message: '✅ Ready to type reply at bottom'
      }).catch(() => { });
    } else {
      console.log('⚠️ Content container not found, using window scroll...');

      // Fallback: scroll window to bottom
      window.scrollTo({
        top: document.documentElement.scrollHeight,
        behavior: 'smooth'
      });

      await sleep(1500);

      console.log('✅ Scrolled window to bottom');
    }

    // Step 4: Type reply
    console.log('Step 4: Typing reply...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '⌨️ Typing reply into message box...'
    }).catch(() => { });

    const typeResult = await typeReply(aiResult.reply);

    if (!typeResult.success) {
      return {
        success: false,
        error: `Failed to type reply: ${typeResult.error}`,
        step: 'type_reply'
      };
    }

    console.log('✅ Reply typed successfully');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'SUCCESS',
      message: '✅ Reply typed successfully'
    }).catch(() => { });

    // Step 5: Click send button
    console.log('Step 5: Clicking send button...');
    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'INFO',
      message: '📤 Clicking send button...'
    }).catch(() => { });

    const sendResult = await clickSendButton();

    if (!sendResult.success) {
      return {
        success: false,
        error: `Failed to click send button: ${sendResult.error}`,
        step: 'click_send'
      };
    }

    console.log('✅ Send button clicked - Message sent!');

    // ⏱️ Calculate response time (from unread click to send button click)
    const endTime = Date.now();
    const responseTimeMs = endTime - startTime;
    // Use Math.ceil to ensure we always get at least 1 second for tracking
    // (Math.floor would give 0 for replies under 1 second, which won't update avgTime)
    const responseTimeSec = Math.ceil(responseTimeMs / 1000);

    console.log(`⏱️ Response time: ${responseTimeSec} seconds (${responseTimeMs}ms)`);
    console.log('=== Read and Reply Flow Complete ===');

    chrome.runtime.sendMessage({
      action: 'addLog',
      type: 'SUCCESS',
      message: `✅ Message sent successfully! (${responseTimeSec}s)`
    }).catch(() => { });

    // 📊 Update analytics after successful reply with actual response time
    console.log(`📊 Updating analytics (response time: ${responseTimeSec}s, customerId: ${userId})...`);
    chrome.runtime.sendMessage({
      action: 'updateAnalyticsAfterReply',
      success: true,
      responseTime: responseTimeSec,  // ✅ Real time from start to send
      customerId: userId  // ✅ Customer ID for unique tracking
    }).catch(err => {
      console.error('Failed to update analytics:', err);
    });

    return {
      success: true,
      messagesRead: readResult.count,
      reply: aiResult.reply,
      message: 'Reply sent successfully',
      // Gig matching info
      gigMatched: gigInfo?.matched || false,
      gigNumber: gigInfo?.gigNumber || null,
      matchedGigTitle: gigInfo?.title || null,
      gigLink: readResult.gigLink || null,
      // Truncation info
      truncated: aiResult.truncated || false,
      messageCount: aiResult.messageCount || readResult.count,
      totalMessages: aiResult.totalMessages || readResult.count
    };

  } catch (error) {
    console.error('Error in readAndReply:', error);

    // ⏱️ Calculate time even on failure
    const endTime = Date.now();
    const responseTimeMs = endTime - startTime;
    const responseTimeSec = Math.floor(responseTimeMs / 1000);

    console.log(`⏱️ Failed after ${responseTimeSec} seconds`);

    // 📊 Update analytics with failure
    console.log(`📊 Recording failed reply attempt (customerId: ${userId})...`);
    chrome.runtime.sendMessage({
      action: 'updateAnalyticsAfterReply',
      success: false,  // ❌ Failed!
      responseTime: 0,  // Don't count failed time in average
      customerId: userId  // ✅ Customer ID (even for failures)
    }).catch(err => {
      console.error('Failed to update analytics:', err);
    });

    return {
      success: false,
      error: error.message,
      step: 'unknown'
    };
  }
}

