// API Client for Backend Communication

const API_URL = 'https://fivreply-production.up.railway.app/api';

class APIClient {
  constructor() {
    this.token = null;
    this.deviceId = null;
    this.deviceName = null;
    this.init();
  }

  async init() {
    await this.loadToken();
    await this.loadDeviceInfo();
  }

  async loadToken() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['authToken'], (result) => {
        this.token = result.authToken || null;
        resolve();
      });
    });
  }

  setToken(token) {
    this.token = token;
    chrome.storage.local.set({ authToken: token });
  }

  clearToken() {
    this.token = null;
    chrome.storage.local.remove('authToken');
  }

  async loadDeviceInfo() {
    this.deviceId = await this.getDeviceId();
    this.deviceName = this.getDeviceName();
  }

  async getDeviceId() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['deviceId'], (result) => {
        if (result.deviceId) {
          resolve(result.deviceId);
        } else {
          // Generate new device ID
          const deviceId = `device_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
          chrome.storage.local.set({ deviceId }, () => {
            resolve(deviceId);
          });
        }
      });
    });
  }

  getDeviceName() {
    const ua = navigator.userAgent;
    let browser = 'Unknown Browser';
    let os = 'Unknown OS';

    // Detect browser
    if (ua.indexOf('Chrome') > -1) browser = 'Chrome';
    else if (ua.indexOf('Firefox') > -1) browser = 'Firefox';
    else if (ua.indexOf('Safari') > -1) browser = 'Safari';
    else if (ua.indexOf('Edge') > -1) browser = 'Edge';

    // Detect OS
    if (ua.indexOf('Win') > -1) os = 'Windows';
    else if (ua.indexOf('Mac') > -1) os = 'Mac';
    else if (ua.indexOf('Linux') > -1) os = 'Linux';

    return `${browser} on ${os}`;
  }

  async request(endpoint, options = {}) {
    const headers = {
      'Content-Type': 'application/json',
      ...options.headers
    };

    if (this.token) {
      headers['Authorization'] = `Bearer ${this.token}`;
    }
    if (this.deviceId) {
      headers['X-Device-Id'] = this.deviceId;
    }

    try {
      // ✅ Add timeout protection (30 seconds)
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 30000);

      const response = await fetch(`${API_URL}${endpoint}`, {
        ...options,
        headers,
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      // ✅ Validate response status
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({ message: 'Unknown error' }));
        throw new Error(errorData.message || `API Error: ${response.status} ${response.statusText}`);
      }

      // ✅ Validate Content-Type
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        throw new Error('Invalid response type: expected JSON');
      }

      // ✅ Parse and validate JSON
      const data = await response.json();

      if (!data || typeof data !== 'object') {
        throw new Error('Invalid response: not a valid JSON object');
      }

      return data;

    } catch (error) {
      // ✅ Handle specific error types
      if (error.name === 'AbortError') {
        console.error('API Error: Request timeout');
        throw new Error('Request timeout - please check your connection');
      }

      if (error.message.includes('Failed to fetch') || error.message.includes('NetworkError')) {
        console.error('API Error: Network error');
        throw new Error('Network error - please check your connection');
      }

      console.error('API Error:', error.message);
      throw error;
    }
  }

  // Authentication APIs
  async register(email, password, extra = {}) {
    if (!this.deviceId || !this.deviceName) {
      await this.loadDeviceInfo();
    }

    const payload = {
      email,
      password,
      fiverrUsername: extra.fiverrUsername,
      deviceId: this.deviceId,
      deviceName: this.deviceName
    };

    const data = await this.request('/auth/register', {
      method: 'POST',
      body: JSON.stringify(payload)
    });
    this.setToken(data.token);
    return data;
  }

  async login(email, password) {
    if (!this.deviceId || !this.deviceName) {
      await this.loadDeviceInfo();
    }

    const data = await this.request('/auth/login', {
      method: 'POST',
      body: JSON.stringify({
        email,
        password,
        deviceId: this.deviceId,
        deviceName: this.deviceName
      })
    });
    this.setToken(data.token);
    return data;
  }

  async getUser() {
    return await this.request('/auth/me');
  }

  async getCurrentUser() {
    return await this.getUser();
  }

  // User Profile APIs
  async updateProfile(fiverrUsername, timeInterval) {
    return await this.request('/user/profile', {
      method: 'PUT',
      body: JSON.stringify({
        fiverrUsername,
        timeInterval,
        deviceId: this.deviceId,
        deviceName: this.deviceName
      })
    });
  }

  async updatePlan(plan) {
    return await this.request('/user/plan', {
      method: 'PUT',
      body: JSON.stringify({ plan })
    });
  }

  async updateBotStatus(status) {
    return await this.request('/user/bot-status', {
      method: 'PUT',
      body: JSON.stringify({ status })
    });
  }

  async updateAnalytics(analytics) {
    return await this.request('/user/analytics', {
      method: 'PUT',
      body: JSON.stringify(analytics)
    });
  }

  // Device Authorization
  async checkDeviceAuth() {
    return await this.request('/user/check-device-auth', {
      method: 'POST',
      body: JSON.stringify({ deviceId: this.deviceId })
    });
  }

  async getDevices() {
    return await this.request('/user/devices');
  }

  // Gigs APIs
  async getGigs() {
    return await this.request('/gigs');
  }

  async getUsernameChangeStatus() {
    return await this.request('/user/username-change-status');
  }

  async saveGigs(gigs) {
    return await this.request('/gigs/bulk', {
      method: 'POST',
      body: JSON.stringify({ gigs })
    });
  }

  async updateGigSummary(gigId, summary) {
    return await this.request(`/gigs/${gigId}/summary`, {
      method: 'PUT',
      body: JSON.stringify({ summary })
    });
  }

  // Analytics APIs
  async getAnalytics() {
    return await this.request('/analytics');
  }

  async recordReply(responseTime = 0, success = true) {
    return await this.request('/analytics/reply', {
      method: 'POST',
      body: JSON.stringify({ responseTime, success })
    });
  }

  async updateActiveChats(count) {
    return await this.request('/analytics/active', {
      method: 'POST',
      body: JSON.stringify({ count })
    });
  }

  async getAnalyticsHistory(days = 7) {
    return await this.request(`/analytics/history?days=${days}`);
  }

  // Plans APIs
  async getPlans() {
    return await this.request('/plans');
  }

  async getAllPlans() {
    return await this.request('/plans/all');
  }

  async getPlan(planId) {
    return await this.request(`/plans/${planId}`);
  }

  async createPlan(planData) {
    return await this.request('/plans', {
      method: 'POST',
      body: JSON.stringify(planData)
    });
  }

  async updatePlan(planId, planData) {
    return await this.request(`/plans/${planId}`, {
      method: 'PUT',
      body: JSON.stringify(planData)
    });
  }

  async deletePlan(planId) {
    return await this.request(`/plans/${planId}`, {
      method: 'DELETE'
    });
  }

  async togglePlan(planId) {
    return await this.request(`/plans/${planId}/toggle`, {
      method: 'PATCH'
    });
  }

  // Logs APIs
  async getLogs(limit = 50) {
    return await this.request(`/logs?limit=${limit}`);
  }

  async createLog(type, message, details = {}) {
    return await this.request('/logs', {
      method: 'POST',
      body: JSON.stringify({ type, message, details })
    });
  }

  // AI APIs (OpenAI through backend)
  async generateAIReply(messages, gigInfo = null, buyerUsername = null, conversationId = null) {
    return await this.request('/ai/generate-reply', {
      method: 'POST',
      body: JSON.stringify({
        messages,
        gigInfo,
        buyerUsername,
        conversationId
      })
    });
  }

  async generateSummary(messages, buyerUsername = null) {
    return await this.request('/ai/generate-summary', {
      method: 'POST',
      body: JSON.stringify({
        messages,
        buyerUsername
      })
    });
  }
}

// Create global instance
const api = new APIClient();

// Make it available globally for content scripts
if (typeof window !== 'undefined') {
  window.api = api;
}

